package org.quidity.demo.util;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ComponentName;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Environment;
import android.telephony.TelephonyManager;
import android.util.Log;
import android.view.WindowManager;
import android.webkit.MimeTypeMap;

import com.github.filosganga.geogson.model.Feature;
import com.google.android.gms.analytics.HitBuilders;
import com.google.android.gms.analytics.Tracker;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.quidity.demo.OneMapApplication;
import org.quidity.demo.Session;
import org.quidity.demo.Settings;

import io.swagger.client.ApiClient;
import io.swagger.client.Configuration;
import io.swagger.client.model.AttachmentsCollection;
import io.swagger.client.model.FeatureWithAttachments;
import io.swagger.client.model.Field;
import io.swagger.client.model.Layer;
import io.swagger.client.model.LineString;
import io.swagger.client.model.Point;
import io.swagger.client.model.Polygon;

/**
 * Created by ferdiedanzfuss on 2014/08/22.
 */
public class AppUtil {

    private static final String TAG = AppUtil.class.getCanonicalName();

    public static final int CONNECTION_TYPE_NONE = 0;
    public static final int CONNECTION_TYPE_CELLULAR= 10;
    public static final int CONNECTION_TYPE_WIFI = 20;
    public static final int CONNECTION_TYPE_OTHER = 100;

    public static String readAsset(Context context, String assetFile) {
        String response = "";
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(
                    new InputStreamReader(context.getAssets().open(assetFile)));

            String line = reader.readLine();
            while (line != null) {
                response += "\n" + line;
                line = reader.readLine();
            }
        } catch (IOException e) {
            Log.e(AppUtil.class.getCanonicalName(), e.getLocalizedMessage(), e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    //ignore the exception
                }
            }
        }
        return response;
    }

    public static void initActivity(Activity activity) {

        Settings settings = Settings.getSettings(activity);
        Session session = Session.getSession();

        //lock device orientation
        //noinspection ResourceType
        activity.setRequestedOrientation(session.getOrientation());

        //keep screen on
        if (settings.isKeepScreenOn()) {
            activity.getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        }
        else {
            activity.getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        }

        //Google Analytics log screen view
        Tracker tracker = ((OneMapApplication)activity.getApplication()).getTracker();
        //tracker.setScreenName(activity.getTitle().toString());
        //tracker.send(new HitBuilders.AppViewBuilder().build());

    }

    // helper function to check if an app is installed
    public static boolean isAppInstalled(Context context, String uri) {
        PackageManager pm = context.getPackageManager();
        boolean app_installed = false;
        try {
            pm.getPackageInfo(uri, PackageManager.GET_ACTIVITIES);
            app_installed = true;
        } catch (PackageManager.NameNotFoundException e) {
            app_installed = false;
        }
        return app_installed;
    }

    public static String pathForAttachment(Context context, long attachmentId, AttachmentsCollection.HeaderEnum headerEnum) {
//        String extension = MimeTypeMap.getSingleton().getExtensionFromMimeType(mimeType);
//        if (extension == null) extension = "unknown";
        String extension = AppUtil.getExtensionForHeader(headerEnum);
        String filename = "Attachment_" + attachmentId + "." + extension;
        File file = new File(context.getCacheDir(), filename);
        return file.getAbsolutePath();
    }

    public static String pathForOfflineMap(Context context, String guid) {
        String filename = guid + ".zip";
        //TODO save on internal storage for production
        String path = context.getFilesDir() + File.separator + "offline";
        //String path = Environment.getExternalStorageDirectory() + File.separator + "offline";
        File folder = new File(path);
        if (!folder.exists()) {
            folder.mkdir();
        }
        File file = new File(folder, filename);

        return file.getAbsolutePath();
    }


    public static long getPrimaryIdForSelectedFeature() {
        Session session = Session.getSession();
        Layer layer = session.getCurrentActiveLayer();
        Feature feature = session.getSelectedFeatures2().get(session.getSelectedFeatureIndex());

        String primaryField = layer.getPrimaryField();

        //String primaryIdStr = (String)feature.getProperties().get(primaryField);
        String primaryIdStr = feature.properties().get(primaryField).getAsString();


        long primaryId = 0;
        if (primaryIdStr != null) {
            primaryId = Long.parseLong(primaryIdStr);
        }

        return primaryId;
    }

    public static long getPrimaryIdForFeature(int featureIndex) {
        Session session = Session.getSession();
        Layer layer = session.getCurrentActiveLayer();
        Feature feature = session.getSelectedFeatures2().get(featureIndex);

        String primaryField = layer.getPrimaryField();

        //String primaryIdStr = (String)feature.getProperties().get(primaryField);
        String primaryIdStr = feature.properties().get(primaryField).getAsString();
        long primaryId = Long.parseLong(primaryIdStr);

        return primaryId;
    }


    public static double dms2dd(int degrees, int minutes, int seconds) {
        double dDegress = (double)degrees;
        double dMinutes = (double)minutes;
        double dSeconds = (double)seconds;

        double dd = dDegress + (dMinutes/60) + (dSeconds/3600);
        return dd;
    }

    public static String captionForField(String fieldName) {
        String caption = fieldName; //default fallback

        List<Field> fields = Session.getSession().getSelectedFeatureFields();
        if(fields != null) {
            for(Field field : fields) {
                if (field.getFieldName().equalsIgnoreCase(fieldName)) {
                    caption = field.getFieldCaption();
                    break;
                }
            }
        }

        return caption;
    }

    public static int currentConnectionType(Context context) {
        int connectionType = CONNECTION_TYPE_NONE;

        ConnectivityManager manager = (ConnectivityManager)context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo info = manager.getActiveNetworkInfo();
        if (info != null && info.isConnected()) {
            int type = info.getType();
            if(type == ConnectivityManager.TYPE_WIFI) {
                connectionType = CONNECTION_TYPE_WIFI;
            }
            else if(type == ConnectivityManager.TYPE_MOBILE) {
                connectionType = CONNECTION_TYPE_CELLULAR;
            }
            else {
                connectionType = CONNECTION_TYPE_OTHER;
            }
        }

        return connectionType;
    }

    public static boolean deleteFile(String filePath) {

        boolean deleted = false;

        try {
            Log.d(AppUtil.class.getCanonicalName(), "Deleting file: " + filePath);
            File file = new File(filePath);
            deleted = file.delete();
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        if(!deleted) {
            Log.w(AppUtil.class.getCanonicalName(), "WARNING: COUNT NOT DELETE FILE: " + filePath);
        }

        return deleted;
    }

    public static void clearAttachmentsCache(Context context) {
        String className = AppUtil.class.getCanonicalName();
        Log.d(className, "Clearing attachments cache.");

        File cacheDir = context.getCacheDir();
        for (File file : cacheDir.listFiles()) {
            boolean deleted = file.delete();

            if (deleted) {
                Log.d(className, "Deleted " + file.getAbsolutePath());
            }
            else {
                Log.d(className, "Could NOT delete " + file.getAbsolutePath());
            }
        }

    }

    public static void promptEnableGPS(final Activity context) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle("Enable GPS");
        builder.setMessage(
                "Your GPS is disabled. Would you like to enable it?")
                .setCancelable(false)
                .setPositiveButton("Yes",
                        new DialogInterface.OnClickListener() {

                            public void onClick(DialogInterface dialog,
                                                int id) {
                                final Intent intent = new Intent(
                                        android.provider.Settings.ACTION_LOCATION_SOURCE_SETTINGS);
                                context.startActivity(intent);
                                dialog.dismiss();
                            }
                        })
                .setNegativeButton("No",
                        new DialogInterface.OnClickListener() {

                            public void onClick(DialogInterface dialog,
                                                int id) {
                                dialog.cancel();
                            }
                        });
        AlertDialog alert = builder.create();
        alert.show();
    }

    //used by offline maps
    public static String getDeviceID(Context context)
    {
        final TelephonyManager tm = (TelephonyManager) context
                .getSystemService(Context.TELEPHONY_SERVICE);

        final String tmDevice, tmSerial, tmPhone, androidId;
        tmDevice = "" + tm.getDeviceId();
        tmSerial = "";// + tm.getSimSerialNumber();
        androidId = ""
                + android.provider.Settings.Secure.getString(
                context.getContentResolver(),
                android.provider.Settings.Secure.ANDROID_ID);

        UUID deviceUuid = new UUID(androidId.hashCode(),
                ((long) tmDevice.hashCode() << 32) | tmSerial.hashCode());
        String deviceId = deviceUuid.toString();

        return deviceId;
    }

    public static void unzip(String filename) {
        try {

            File srcFile = new File(filename);
            String destPath = srcFile.getParentFile().getAbsolutePath() + File.separator + srcFile.getName().replaceAll(".zip", "");

            ZipFile zip = new ZipFile(filename);

            Enumeration<? extends ZipEntry> zippedFiles = zip.entries();
            while (zippedFiles.hasMoreElements()) {
                ZipEntry entry = zippedFiles.nextElement();
                InputStream is = zip.getInputStream(entry);
                String name = entry.getName();
                File outputFile = new File(destPath + File.separator + name);
                String outputPath = outputFile.getCanonicalPath();
                name = outputPath.substring(outputPath.lastIndexOf("/") + 1);
                outputPath = outputPath.substring(0, outputPath.lastIndexOf("/"));
                File outputDir = new File(outputPath);
                outputDir.mkdirs();
                outputFile = new File(outputPath, name);
                outputFile.createNewFile();
                FileOutputStream out = new FileOutputStream(outputFile);
                byte buf[] = new byte[16384];
                do {
                    int numread = is.read(buf);
                    if (numread <= 0) {
                        break;
                    } else {
                        out.write(buf, 0, numread);
                    }
                } while (true);

                is.close();
                out.close();
            }
            File theZipFile = new File(filename);
            theZipFile.delete();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    public static ApiClient getApiClient() {
        ApiClient client = Configuration.getDefaultApiClient();
        client.setBasePath(Settings.API_BASE_URL);
        return client;
    }

    private static final String EXT_TXT = "txt";
    private static final String EXT_3GPP = "3gpp";
    private static final String EXT_JPG = "jpg";
    private static final String EXT_MP4 = "mp4";

    public static AttachmentsCollection.HeaderEnum getHeaderForExtension(String extension) {
        if (extension.equalsIgnoreCase(EXT_TXT)) {
            return AttachmentsCollection.HeaderEnum.TEXT_PLAIN;
        }
        else if (extension.equalsIgnoreCase(EXT_3GPP)) {
            return AttachmentsCollection.HeaderEnum.AUDIO_3GPP;
        }
        else if (extension.equalsIgnoreCase(EXT_JPG)) {
            return AttachmentsCollection.HeaderEnum.IMAGE_JPEG;
        }
        else if (extension.equalsIgnoreCase(EXT_MP4)) {
            return AttachmentsCollection.HeaderEnum.VIDEO_MP4;
        }
        else {
            Log.w(TAG, "Unknown extension " + extension);
            return null;
        }
    }

    public static String getExtensionForHeader(AttachmentsCollection.HeaderEnum header) {

        if (header == AttachmentsCollection.HeaderEnum.TEXT_PLAIN) {
            return EXT_TXT;
        }
        else if (header == AttachmentsCollection.HeaderEnum.AUDIO_3GPP) {
            return EXT_3GPP;
        }
        else if (header == AttachmentsCollection.HeaderEnum.IMAGE_JPEG) {
            return EXT_JPG;
        }
        else if (header == AttachmentsCollection.HeaderEnum.VIDEO_MP4) {
            return EXT_MP4;
        }
        else {
            Log.w(TAG, "Unknown header " + header.toString());
            return null;
        }


    }


}
