package org.quidity.demo.helper;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Bundle;
import android.util.Log;
import android.widget.ArrayAdapter;
import android.widget.ListAdapter;
import android.widget.Toast;

import com.github.filosganga.geogson.model.Feature;
import com.google.gson.Gson;

import org.quidity.demo.Session;
import org.quidity.demo.Settings;
import org.quidity.demo.db.AttachmentEntry;
import org.quidity.demo.db.AttachmentsDAO;
import org.quidity.demo.db.OfflineUpdateEntry;
import org.quidity.demo.db.OfflineUpdatesDAO;
import org.quidity.demo.util.AppUtil;
import org.quidity.demo.util.GuiUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import io.swagger.client.ApiCallback;
import io.swagger.client.ApiException;
import io.swagger.client.api.AttributesBasicApi;
import io.swagger.client.api.SpatialApi;
import io.swagger.client.model.FeatureBasic;
import io.swagger.client.model.FeatureWithoutPropertiesAndAttachments;
import io.swagger.client.model.Geometry;
import io.swagger.client.model.OKResponseAttachmentsUpdate;
import io.swagger.client.model.OKResponseAttributesAdd;
import io.swagger.client.model.OKResponseAttributesAddResult;
import io.swagger.client.model.OKResponseAttributesUpdate;
import io.swagger.client.model.OKResponseAttributesUpdateResult;

/**
 * Created by ferdiedanzfuss on 2016/04/19.
 */
public class OfflineUpdatesSyncHelper {

    private final static String TAG = OfflineUpdatesSyncHelper.class.getCanonicalName();

    private final static int UPLOAD_RESULT_SUCCESS = 10;
    private final static int UPLOAD_RESULT_FAILED = 20;

    private Activity context;
    private OfflineUpdatesSyncListener listener;

    private List<OfflineUpdateEntry> entries;
    private ProgressDialog progressDialog;
    private BlockingQueue<Integer> resultQueue;

    private int successCount = 0;
    private int failCount = 0;

    private long currentRowId;


    public OfflineUpdatesSyncHelper(Activity context) {
        this.context = context;
    }

    public void startOfflineUpdatesSync(OfflineUpdatesSyncListener listener) {
        this.listener = listener;

        this.entries = OfflineUpdatesDAO.getAllEntries(this.context);

        checkConnectivityBeforeUpload();
    }


    private void checkConnectivityBeforeUpload() {

        if(AppUtil.currentConnectionType(context) == AppUtil.CONNECTION_TYPE_WIFI) {
            uploadAll();
        }
        else {

            AlertDialog.Builder builder = new AlertDialog.Builder(context);
            builder.setTitle("WiFi not available");
            builder.setMessage("A WiFi network is not currently available. Do you want to continue on cellular?");
            builder.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialogInterface, int i) {
                    dialogInterface.dismiss();
                    uploadAll();
                }
            });
            builder.setNegativeButton("No", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialogInterface, int i) {
                    //go back
                    OfflineUpdatesSyncHelper.this.listener.offlineUpdatesSynCompleted();
                }
            });
            builder.create().show();

        }

    }

    private void uploadAll() {

        if (entries.size() > 0) {

            progressDialog = new ProgressDialog(context);
            progressDialog.setTitle("Syncing Offline Updates");
            progressDialog.show();

            Thread uploadThread = new Thread(new SyncWorker());
            uploadThread.start();

        }
        else {
            this.listener.offlineUpdatesSynCompleted();
        }
    }

    class SyncWorker implements Runnable

    {

        @Override
        public void run() {
            //process 1 entry at a time.
            resultQueue = new ArrayBlockingQueue<Integer>(1);

            successCount = 0;
            failCount = 0;
            List<OfflineUpdateEntry> entriesToUpload = new ArrayList<OfflineUpdateEntry>(entries);
            int entryCount = entriesToUpload.size();

            for(int c=0;c<entryCount;c++) {
                OfflineUpdateEntry entry = entriesToUpload.get(c);
                currentRowId = entry.getRowId();

                final int entryNo = c+1;

                final String message = "Sync " + entryNo + " of " + entryCount + "...";

                context.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        progressDialog.setMessage(message);
                    }
                });

                if (entry.getUpdateType().equalsIgnoreCase(OfflineUpdateEntry.UPdATE_TYPE_EDIT_FEATURE)) {
                    performEditFeature(entry);
                }
                else if (entry.getUpdateType().equalsIgnoreCase(OfflineUpdateEntry.UPdATE_TYPE_ADD_FEATURE)) {
                    performAddFeature(entry);
                }
                else if (entry.getUpdateType().equalsIgnoreCase(OfflineUpdateEntry.UPdATE_TYPE_EDIT_GEOMETRY)) {
                    performEditGeometry(entry);
                }
                else {
                    Log.e(getClass().getCanonicalName(), "Unsupported update type: " + entry.getUpdateType());
                }


                try {
                    //the queue will block until we have a result
                    int result = resultQueue.take();

                    if (result ==  UPLOAD_RESULT_SUCCESS) {
                        successCount++;
                        OfflineUpdatesDAO.deleteEntry(context, entry.getRowId());
                    }
                    else {
                        failCount++;
                        Toast.makeText(context, "Sync failed for row " + entry.getRowId(), Toast.LENGTH_LONG).show();
                    }
                }
                catch (InterruptedException e) {
                    e.printStackTrace();
                }



            }


            //all done
            final int finalSuccessCount = successCount;
            final int finalFailCount = failCount;
            context.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    progressDialog.dismiss();

                    //Show results dialog message
                    AlertDialog.Builder builder = new AlertDialog.Builder(context);
                    builder.setTitle("Sync Completed");
                    String message = "Success: " + finalSuccessCount + ", Failed: " + finalFailCount;
                    builder.setMessage(message);
                    builder.setPositiveButton("OK", new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialogInterface, int i) {
                            dialogInterface.dismiss();

                            OfflineUpdatesSyncHelper.this.listener.offlineUpdatesSynCompleted();
                        }
                    });
                    builder.create().show();
                }
            });

        }

        private void performEditFeature(OfflineUpdateEntry entry) {

            Log.d(getClass().getCanonicalName(), "performEditFeature");

//            Gson gson = new Gson();
//            EditAttributesRequest request = gson.fromJson(entry.getRequestJson(), EditAttributesRequest.class);
//            Feature feature = gson.fromJson(entry.getObjectJson(), Feature.class);
//
//            AttributesService service = new AttributesService();
//            service.editAttributes(request, feature, this);


            try {
                AttributesBasicApi api = new AttributesBasicApi(AppUtil.getApiClient());

                long layerId = entry.getLayerId();
                long primaryId = entry.getPrimaryId();
                //TODO create feature object
                FeatureBasic feature = new FeatureBasic();

                api.attributesPrimaryIdEditAsync(layerId, primaryId, feature, new ApiCallback<OKResponseAttributesUpdate>() {
                    @Override
                    public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                        editAttributesFailed(e.getMessage());
                    }

                    @Override
                    public void onSuccess(OKResponseAttributesUpdate result, int statusCode, Map<String, List<String>> responseHeaders) {
                        editAttributesCompleted(result.getResult());
                    }

                    @Override
                    public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {

                    }

                    @Override
                    public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {

                    }
                });
            }
            catch (ApiException e) {
                Log.e(TAG, e.getMessage(), e);
                GuiUtil.showWarning(context, e.getMessage());
            }
        }


        private void performAddFeature(OfflineUpdateEntry entry) {

            Log.d(getClass().getCanonicalName(), "performAddFeature");

//            AttributesService service = new AttributesService();
//
//            Gson gson = new Gson();
//            AddAttributesRequest request = gson.fromJson(entry.getRequestJson(), AddAttributesRequest.class);
//            Feature feature = gson.fromJson(entry.getObjectJson(), Feature.class);
//            service.addAttributes(request, feature, this);

            try {
                AttributesBasicApi api = new AttributesBasicApi(AppUtil.getApiClient());
                long layerId = entry.getLayerId();
                //TODO populate feature
                FeatureBasic feature = new FeatureBasic();
                api.attributesPrimaryIdAddAsync(layerId, feature, new ApiCallback<OKResponseAttributesAdd>() {
                    @Override
                    public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                        addAttributesFailed(e.getMessage());
                    }

                    @Override
                    public void onSuccess(OKResponseAttributesAdd result, int statusCode, Map<String, List<String>> responseHeaders) {
                        addAttributesCompleted(result.getResult());
                    }

                    @Override
                    public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {

                    }

                    @Override
                    public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {

                    }
                });
            }
            catch (ApiException e) {
                Log.e(TAG, e.getMessage(), e);
                GuiUtil.showWarning(context, e.getMessage());
            }
        }

        private void performEditGeometry(OfflineUpdateEntry entry) {

            Log.d(getClass().getCanonicalName(), "performEditGeometry");

            Gson gson = new Gson();
            //UpdateFeatureRequest request = gson.fromJson(entry.getRequestJson(), UpdateFeatureRequest.class);
            long layerId = entry.getLayerId();
            long primaryId = entry.getPrimaryId();
            Geometry geometry = gson.fromJson(entry.getObjectJson(), Geometry.class);

//            SpatialService service = new SpatialService();
//            service.updateFeature(request, geometry, this);
            try {
                SpatialApi api = new SpatialApi(AppUtil.getApiClient());

                FeatureWithoutPropertiesAndAttachments feature = new FeatureWithoutPropertiesAndAttachments();
                feature.setGeometry(geometry);
                feature.setType(FeatureWithoutPropertiesAndAttachments.TypeEnum.FEATURE);

                //TODO
                Feature fixmeFeature = null;

                api.apiSpatialUpdateGeometryAsync(layerId, primaryId, fixmeFeature, new ApiCallback<OKResponseAttachmentsUpdate>() {
                    @Override
                    public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                        updateFeatureFailed(e.getMessage());
                    }

                    @Override
                    public void onSuccess(OKResponseAttachmentsUpdate result, int statusCode, Map<String, List<String>> responseHeaders) {
                        updateFeatureCompleted(result);
                    }

                    @Override
                    public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {

                    }

                    @Override
                    public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {

                    }
                });
            }
            catch (ApiException e) {
                Log.e(TAG, e.getMessage(), e);
                GuiUtil.showWarning(context, e.getMessage());
            }

        }


        private void updatePrimaryIdForLinkedGeometryEntries(long primaryId) {
            Gson gson = new Gson();

            for (OfflineUpdateEntry entry : entries) {
                if (entry.getUpdateType().equalsIgnoreCase(OfflineUpdateEntry.UPdATE_TYPE_EDIT_GEOMETRY)) {
                    if (entry.getLinkedRowId() == currentRowId) {
//                        UpdateFeatureRequest request = gson.fromJson(entry.getRequestJson(), UpdateFeatureRequest.class);
//                        request.setPrimaryId(primaryId);
//                        entry.setRequestJson(gson.toJson(request));
                        entry.setPrimaryId(primaryId);

                        //TODO perhaps also update the db
                        //OfflineUpdatesDAO.updateEntry(this, entry);
                    }
                }
            }

        }

        private void updatePrimaryIdForLinkedEditFeatureEntries(long primaryId) {
            //TODO foigure out how to do this
//            Gson gson = new Gson();
//
//            for (OfflineUpdateEntry entry : entries) {
//                if (entry.getUpdateType().equalsIgnoreCase(OfflineUpdateEntry.UPdATE_TYPE_EDIT_FEATURE)) {
//                    if (entry.getLinkedRowId() == currentRowId) {
//                        EditAttributesRequest request = gson.fromJson(entry.getRequestJson(), EditAttributesRequest.class);
//                        request.setPrimaryId(primaryId);
//
//                        Feature feature = gson.fromJson(entry.getObjectJson(), Feature.class);
//                        //TODO onemapfid primary key name should be determined dynamically
//                        feature.getProperties().put("onemapfid", Long.toString(primaryId));
//                        feature.getProperties().remove("offline_row_id");
//
//                        entry.setRequestJson(gson.toJson(request));
//                        entry.setObjectJson(gson.toJson(feature));
//                        //TODO perhaps also update the db
//                        //OfflineUpdatesDAO.updateEntry(this, entry);
//                    }
//                }
//            }

        }

        private void updatePrimaryIdForLinkedAttachments(long primaryId) {
            Gson gson = new Gson();

            Context context = OfflineUpdatesSyncHelper.this.context;
            List<AttachmentEntry> attEntries = AttachmentsDAO.getAllEntries(context);

            for (AttachmentEntry entry : attEntries) {

                    if (entry.getOfflineUpdateRowId() == currentRowId) {
                        //entry.getRequest().setPrimaryId(primaryId);
                        entry.setPrimaryId(primaryId);

                        //update the db
                        AttachmentsDAO.deleteEntry(context, entry.getRowId());
                        AttachmentsDAO.insertEntry(context, entry);
                    }

            }

        }




        public void editAttributesCompleted(OKResponseAttributesUpdateResult response) {
            try {
                resultQueue.put(UPLOAD_RESULT_SUCCESS);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }


        public void editAttributesFailed(String reason) {
            try {
                resultQueue.put(UPLOAD_RESULT_FAILED);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }


        public void addAttributesCompleted(OKResponseAttributesAddResult response) {
            try {
                long primaryId = response.getPrimaryId();
                updatePrimaryIdForLinkedGeometryEntries(primaryId);
                updatePrimaryIdForLinkedEditFeatureEntries(primaryId);
                updatePrimaryIdForLinkedAttachments(primaryId);

                //TODO update linked attachments

                resultQueue.put(UPLOAD_RESULT_SUCCESS);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }

        }


        public void addAttributesFailed(String reason) {

            try {
                resultQueue.put(UPLOAD_RESULT_FAILED);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }

        }


        public void updateFeatureCompleted(OKResponseAttachmentsUpdate response) {
            try {
                resultQueue.put(UPLOAD_RESULT_SUCCESS);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }

        }


        public void updateFeatureFailed(String reason) {
            try {
                resultQueue.put(UPLOAD_RESULT_FAILED);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }

        }
    }

}
