package org.quidity.demo.activity;

import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;

import com.google.gson.Gson;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.quidity.demo.R;
import org.quidity.demo.Session;
import org.quidity.demo.Settings;
import org.quidity.demo.util.AppUtil;
import org.quidity.demo.util.GuiUtil;

import io.swagger.client.ApiCallback;
import io.swagger.client.ApiException;
import io.swagger.client.api.AddressesApi;
import io.swagger.client.model.ErfWithNADAddresses;
import io.swagger.client.model.OKResponseAddressSearch;
import io.swagger.client.model.OKResponseAddressSearchResult;
import io.swagger.client.model.OKResponseAddressStandNo;
import io.swagger.client.model.OKResponseAddressStandNoResult;

public class SearchActivity extends Activity {

    private static final String TAG = SearchActivity.class.getCanonicalName();

    private ViewGroup addressLayout;
    private ViewGroup erfLayout;

    private Spinner searchTypeSpinner;
    private EditText addressField;
    private Button searchButton;

    private EditText cityField;
    private EditText erfField;
    private EditText portionField;

    public static final String SEARCH_TYPE_STREET_ADDRESS = "Street Address";
    public static final String SEARCH_TYPE_ERF_NO = "Erf No";
    private List<String> searchTypes;
    private String currentSearchType = SEARCH_TYPE_STREET_ADDRESS;  //default

    private ProgressDialog progressDialog;

    private static final int REQ_SEARCH_RESULTS = 100;

    public SearchActivity() {
        searchTypes = new ArrayList<String>();
        searchTypes.add(SEARCH_TYPE_STREET_ADDRESS);
        searchTypes.add(SEARCH_TYPE_ERF_NO);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_search);

        AppUtil.initActivity(this);

        //restore session state
        if(savedInstanceState != null) {
            Session savedSession = (Session)savedInstanceState.getSerializable(Settings.SESSION_KEY);
            Session.restore(savedSession);
            //Session.restoreSessionState();
        }


        addressLayout = (ViewGroup)findViewById(R.id.address_layout);
        erfLayout = (ViewGroup)findViewById(R.id.erf_layout);

        searchTypeSpinner = (Spinner)findViewById(R.id.search_type);
        addressField = (EditText)findViewById(R.id.address_field);
        searchButton = (Button)findViewById(R.id.search_button);

        cityField = (EditText)findViewById(R.id.city_field);
        erfField = (EditText)findViewById(R.id.erf_field);
        portionField = (EditText)findViewById(R.id.portion_field);

        searchButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                Log.d(TAG, "Clicked on search");
                search();
            }
        });

        ArrayAdapter<String> adapter = new ArrayAdapter<String>(this, android.R.layout.simple_spinner_dropdown_item, searchTypes);
        searchTypeSpinner.setAdapter(adapter);
        searchTypeSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                searchTypeChanged(searchTypes.get(position));
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {

            }
        });

        String searchType = getIntent().getStringExtra("SearchType");
        searchTypeChanged(searchType);
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        //outState.putSerializable(Settings.SESSION_KEY, Session.getSession());
        //Session.saveSessionState();
        super.onSaveInstanceState(outState);
    }

    private void searchTypeChanged(String searchType) {

        addressLayout.setVisibility(View.GONE);
        erfLayout.setVisibility(View.GONE);

        if (searchType.equalsIgnoreCase(SEARCH_TYPE_STREET_ADDRESS)) {
            addressLayout.setVisibility(View.VISIBLE);
            GuiUtil.focus(this, addressField);
        }
        else if(searchType.equalsIgnoreCase(SEARCH_TYPE_ERF_NO)) {
            erfLayout.setVisibility(View.VISIBLE);
            GuiUtil.focus(this, cityField);
        }

        currentSearchType = searchType;
    }

    private void search() {

        Log.d(TAG, "Performing search");

        if (currentSearchType.equalsIgnoreCase(SEARCH_TYPE_STREET_ADDRESS)) {
            String address = addressField.getText().toString();
            if (address.trim().length() == 0) {
                GuiUtil.showWarning(this, "Please enter an address.");
                return;
            }

            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    progressDialog = ProgressDialog.show(SearchActivity.this, "Searching", "Please wait...", true);
                }
            });

//            LocationService service = new LocationService();
//            SearchAddressRequest request = new SearchAddressRequest();
//            request.setAddress(address);
//            request.setStart(0);
//            request.setLimit(50);
//            service.searchAddress(request, this);

            try {
                AddressesApi api = new AddressesApi(AppUtil.getApiClient());
                api.addressSearchAsync(address, 1L, 50L, new ApiCallback<OKResponseAddressSearch>() {
                    @Override
                    public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                        onSearchAddressFailed(e.getMessage());
                    }

                    @Override
                    public void onSuccess(OKResponseAddressSearch result, int statusCode, Map<String, List<String>> responseHeaders) {
                        onSearchAddressCompleted(result.getResult());
                    }

                    @Override
                    public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {

                    }

                    @Override
                    public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {

                    }
                });
            }
            catch (ApiException e) {
                Log.e(TAG, e.getMessage(), e);
                GuiUtil.showWarning(this, e.getMessage());
            }
        }
        else if (currentSearchType.equalsIgnoreCase(SEARCH_TYPE_ERF_NO)) {
            String city = cityField.getText().toString();
            String erfNo = erfField.getText().toString();
            String portionNo = portionField.getText().toString();

            if (city.trim().length() == 0) {
                GuiUtil.showWarning(this, "Please enter a city or township.");
                return;
            }
            if (erfNo.trim().length() == 0) {
                GuiUtil.showWarning(this, "Please enter erf no.");
                return;
            }

            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    progressDialog = ProgressDialog.show(SearchActivity.this, "Searching", "Please wait...", true);
                }
            });


            String searchString = city + " " + erfNo;

            if(portionNo.trim().equalsIgnoreCase("0")) {
                portionNo = "";
            }

//            LocationService service = new LocationService();
//            SearchErfRequest request = new SearchErfRequest();
//            request.setSearchString(searchString);
//            if (portionNo.trim().length() > 0) {
//                request.setPortionNo(portionNo);
//            }
//            request.setStart(0);
//            request.setLimit(50);
//            service.searchErf(request, this);

            try {
                AddressesApi api = new AddressesApi(AppUtil.getApiClient());
                api.addressStandSearchAsync(city, erfNo, portionNo, new ApiCallback<OKResponseAddressStandNo>() {
                    @Override
                    public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                        onSearchErfFailed(e.getMessage());
                    }

                    @Override
                    public void onSuccess(OKResponseAddressStandNo result, int statusCode, Map<String, List<String>> responseHeaders) {
                        onSearchErfCompleted(result.getResult());
                    }

                    @Override
                    public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {

                    }

                    @Override
                    public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {

                    }
                });
            }
            catch (ApiException e) {
                Log.e(TAG, e.getMessage(), e);
                GuiUtil.showWarning(this, e.getMessage());
            }
        }
    }



    public void onSearchAddressCompleted(OKResponseAddressSearchResult response) {

        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressDialog.dismiss();
            }
        });


        Intent intent = new Intent(this, SearchResultsActivity.class);

//        //serialize the response items so we can pass them to the results activity
//        Gson gson = new Gson();
//        String itemsStr = gson.toJson(response.getNadAddresses());
//        intent.putExtra("items", itemsStr);
        Session.getSession().setSearchTypeAddress(true);
        Session.getSession().setNadAddresses(response.getNadAddresses());

        //intent.setFlags(Intent.FLAG_ACTIVITY_FORWARD_RESULT);

        startActivityForResult(intent, REQ_SEARCH_RESULTS);
        //startActivity(intent);

        //Toast.makeText(this, "Under construction. Number of results " + response.getItems().size(), Toast.LENGTH_SHORT);
    }


    public void onSearchAddressFailed(final String reason) {

        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressDialog.dismiss();
                GuiUtil.showError(SearchActivity.this, reason);
            }
        });


    }


    public void onSearchErfCompleted(OKResponseAddressStandNoResult response) {

        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressDialog.dismiss();
            }
        });


        Intent intent = new Intent(this, SearchResultsActivity.class);

//        //serialize the response items so we can pass them to the results activity
//        Gson gson = new Gson();
//        String itemsStr = gson.toJson(response.getStand());
//        intent.putExtra("items", itemsStr);
        Session.getSession().setSearchTypeAddress(false);
        Session.getSession().setErfWithNADAddresses(response.getStand());

        //startActivity(intent);
        startActivityForResult(intent, REQ_SEARCH_RESULTS);
    }


    public void onSearchErfFailed(String reason) {

        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressDialog.dismiss();
            }
        });

        GuiUtil.showError(this, reason);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == REQ_SEARCH_RESULTS) {
            if (resultCode == Activity.RESULT_OK) {
                setResult(Activity.RESULT_OK, data);
                finish();   //will forward result to map activity
            }
        }
    }
}
