package org.quidity.demo.activity;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.ListActivity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ListAdapter;
import android.widget.PopupMenu;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.apache.commons.io.FileUtils;
import org.quidity.demo.R;
import org.quidity.demo.Session;
import org.quidity.demo.Settings;
import org.quidity.demo.db.OfflineMapEntry;
import org.quidity.demo.db.OfflineMapsDAO;
import org.quidity.demo.mapi.offline.DownloadRequest;
import org.quidity.demo.mapi.offline.DownloadResponse;
import org.quidity.demo.mapi.offline.OfflineService;
import org.quidity.demo.mapi.offline.StatusRequest;
import org.quidity.demo.mapi.offline.StatusResponse;
import org.quidity.demo.util.AppUtil;
import org.quidity.demo.util.GuiUtil;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

public class OfflineMapsActivity extends ListActivity implements
        View.OnClickListener,
        AdapterView.OnItemClickListener,
        AdapterView.OnItemLongClickListener,
        OfflineService.StatusListener,
        OfflineService.DownloadListener
{

    private static final int REQUEST_NEW_OFFLINE_MAP = 10;

    private static final int REFRESH_INTERVAL = 2000;   //2 seconds

    private ImageButton addMapButton;

    private List<OfflineMapEntry> entries;
    private OfflineMapEntry refreshEntry;    //item whose status needs to be refreshed

    private OfflineMapsAdapter adapter;

    private ProgressDialog progressDialog;

    private Runnable refreshRunnable;
    private Handler refreshHandler;
    private boolean isActivityRunning;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_offline_maps);

        AppUtil.initActivity(this);

        //restore session state
        if(savedInstanceState != null) {
            Session savedSession = (Session)savedInstanceState.getSerializable(Settings.SESSION_KEY);
            Session.restore(savedSession);
            //Session.restoreSessionState();
        }

        addMapButton = (ImageButton)findViewById(R.id.add_map);
        addMapButton.setOnClickListener(this);

        getListView().setOnItemClickListener(this);
        getListView().setOnItemLongClickListener(this);

    }

    @Override
    protected void onStart() {
        super.onStart();
        Log.d(getClass().getCanonicalName(), "onStart");

        isActivityRunning = true;
        refreshList();
        refreshItemStatus();

    }

    @Override
    protected void onStop() {
        super.onStop();
        Log.d(getClass().getCanonicalName(), "onStop");

        /*
        if (refreshHandler != null && refreshRunnable != null) {
            refreshHandler.removeCallbacks(refreshRunnable);
        }
        */
        isActivityRunning = false;
        if (refreshHandler != null) {
            refreshHandler.removeCallbacksAndMessages(null);
            Log.d(getClass().getCanonicalName(), "Removing all callbacks and messages.");
        }
    }

    private void refreshList() {

        entries = OfflineMapsDAO.getAllEntries(this);

        adapter = new OfflineMapsAdapter(this, entries);
        getListView().setAdapter(adapter);
    }

    /**
     * Check if there is any items whose status needs to be refreshed (who are not yet READY)
     * Handle only one item at a time (the first item)
     */
    private void refreshItemStatus() {

        for (OfflineMapEntry entry : entries) {
            if (!entry.isPrepared()) {
                this.refreshEntry = entry;
                refreshStatus();
                break;
            }
        }

    }



    @Override
    public void onClick(View view) {
        if (view == addMapButton) {
            Intent intent = new Intent(this, NewOfflineMapActivity.class);
            startActivityForResult(intent, REQUEST_NEW_OFFLINE_MAP);
        }
    }

    @Override
    public void onItemClick(AdapterView<?> adapterView, View view, int position, long id) {
        /*
        OfflineMapEntry entry = (OfflineMapEntry)adapterView.getItemAtPosition(position);


        Intent intent = new Intent(this, OfflineMapDetailActivity.class);
        intent.putExtra(OfflineMapDetailActivity.ROW_ID_KEY, entry.getRowId());
        startActivity(intent);
        */
    }

    @Override
    public boolean onItemLongClick(final AdapterView<?> adapterView, View view, final int position, long id) {
        PopupMenu popup = new PopupMenu(this, view);
        popup.getMenuInflater().inflate(R.menu.menu_offlinemaps_item_popup, popup.getMenu());

        popup.setOnMenuItemClickListener(new PopupMenu.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                if (menuItem.getItemId() == R.id.action_offlinemap_delete) {
                    OfflineMapEntry entry = (OfflineMapEntry) adapterView.getItemAtPosition(position);
                    promptDelete(entry);

                    return true;
                }

                return false;
            }
        });

        popup.show();

        return true;

    }

    private void promptDelete(final OfflineMapEntry entry) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage("Are you sure you want to delete " + entry.getName() + "?");
        builder.setNegativeButton("Cancel", null);
        builder.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                delete(entry);
            }
        });
        builder.create().show();
    }

    private void delete(OfflineMapEntry entry) {
        try {
            //try delete the zip file if it exists
            String zipPath = AppUtil.pathForOfflineMap(this, entry.getGuid());
            Log.d(getClass().getCanonicalName(), zipPath);
            forceDelete(zipPath);

            //try delete the unzipped folder if it exists
            File zipFile = new File(zipPath);
            String unzippedPath = zipFile.getParentFile().getAbsolutePath() + File.separator + zipFile.getName().replaceAll(".zip", "");
            forceDelete(unzippedPath);

            OfflineMapsDAO.deleteEntry(this, entry.getRowId());

            Toast.makeText(this, entry.getName() + " deleted.", Toast.LENGTH_SHORT).show();

            refreshList();
        }
        catch(Exception e) {
            e.printStackTrace();
            Toast.makeText(this, "Could not delete. " + e.getLocalizedMessage(), Toast.LENGTH_LONG).show();
        }
    }

    private void forceDelete(String path) {
        try {
            FileUtils.forceDelete(new File(path));
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void setProgressStatus(String mapGuid, String status) {
        for (OfflineMapEntry entry : entries) {
            if (entry.getGuid().equalsIgnoreCase(mapGuid)) {
                entry.setProgressStatus(status);
                adapter.notifyDataSetChanged();
                break;
            }
        }
    }


    private void refreshStatus() {
        Log.d(getClass().getCanonicalName(), "Refreshing status for " + refreshEntry.getName());

        //progressDialog = ProgressDialog.show(this, null, "Please wait...", true);

        StatusRequest request = new StatusRequest();
        request.setGuid(refreshEntry.getGuid());

        OfflineService service = new OfflineService();
        service.status(request, this);

    }

    private void download() {

        progressDialog = new ProgressDialog (this);
        progressDialog.setTitle("Please wait");
        progressDialog.setMessage("Downloading...");
        progressDialog.setIndeterminate(false);
        progressDialog.setProgressStyle(ProgressDialog.STYLE_HORIZONTAL);
        progressDialog.setProgress(0);
        progressDialog.setMax(100);
        progressDialog.show();

        //statusView.setText("DOWNLOADING");
        setProgressStatus(refreshEntry.getGuid(), "Downloading");
        //progressView.setProgress(0);

        DownloadRequest request = new DownloadRequest();
        request.setGuid(refreshEntry.getGuid());

        String savePath = AppUtil.pathForOfflineMap(this, refreshEntry.getGuid());
        Log.d(getClass().getCanonicalName(), savePath);

        OfflineService service = new OfflineService();
        service.download(request, savePath, this);
    }

    private void unzip() {
        progressDialog = ProgressDialog.show(this, "Please wait", "Extracting...", true);

        //statusView.setText("PROCESSING");
        setProgressStatus(refreshEntry.getGuid(), "Extracting...");


        String savePath = AppUtil.pathForOfflineMap(this, refreshEntry.getGuid());
        AppUtil.unzip(savePath);

        //flag that the map is now prepared/ready
        refreshEntry.setPrepared(true);
        OfflineMapsDAO.updateEntry(this, refreshEntry);

        //statusView.setText("READY");
        progressDialog.dismiss();
        setProgressStatus(refreshEntry.getGuid(), "Unzipped");
    }

    private void processActiveLayerGeoJson() {
        progressDialog = ProgressDialog.show(this, "Please wait", "Processing...", true);

        try {
            String savePath = AppUtil.pathForOfflineMap(this, refreshEntry.getGuid());
            savePath = savePath.replaceAll(".zip", "");

            String geoJsonFile = savePath + "/activelayer.geojson";
            String geoJsonOnlyFile = savePath + "/activelayer.geojsononly";
            String fieldsFile = savePath + "/activelayer.fields";

            JsonParser parser = new JsonParser();
            JsonElement el = parser.parse(new FileReader(geoJsonFile));
            JsonObject oItems = el.getAsJsonObject().getAsJsonObject("items");
            String geoJsonOnlyStr = oItems.toString();
            Log.d(getClass().getCanonicalName(), geoJsonOnlyStr);

            JsonArray aFields = el.getAsJsonObject().getAsJsonArray("fields");
            String fieldsStr = aFields.toString();
            Log.d(getClass().getCanonicalName(), fieldsStr);

            FileWriter writer = new FileWriter(geoJsonOnlyFile);
            writer.write(geoJsonOnlyStr);
            writer.flush();
            writer.close();

            FileWriter fieldsWriter = new FileWriter((fieldsFile));
            fieldsWriter.write(fieldsStr);
            fieldsWriter.flush();
            fieldsWriter.close();
        }
        catch (Exception e) {
            e.printStackTrace();
        }

        progressDialog.dismiss();
        setProgressStatus(refreshEntry.getGuid(), "Ready");

    }

    @Override
    public void statusCompleted(StatusResponse response) {

        if (!isActivityRunning) {
            return;
        }

        //progressDialog.dismiss();

        //statusView.setText("Status: " + response.getStatus().getStatus());
        String subMessage = null;
        if (response.getStatus().getStatus().equalsIgnoreCase("PENDING")) {
            subMessage = "Pending...";
        }
        else {
            subMessage = response.getStatus().getTilesPrepared() + " of " + response.getStatus().getTotalTiles() + " tiles prepared.";
        }

        //subStatusView.setText(subMessage);
        setProgressStatus(refreshEntry.getGuid(), subMessage);


        //int iProgress = (int)response.getStatus().getPercentageDone();
        //progressView.setProgress(iProgress);

        if (response.getStatus().getStatus().trim().equalsIgnoreCase("Done")) {
            download();
        }
        else {
            //schedule refresh timer
            this.refreshHandler = new Handler();
            this.refreshRunnable = new Runnable() {
                @Override
                public void run() {
                    refreshStatus();
                }
            };
            this.refreshHandler.postDelayed(this.refreshRunnable, REFRESH_INTERVAL);

        }

    }

    @Override
    public void statusFailed(String reason) {
        //progressDialog.dismiss();
        GuiUtil.showError(this, reason);
    }

    @Override
    public void downloadCompleted(DownloadResponse response) {

        //progressView.setProgress(100);
        //statusView.setText("DOWNLOADED");
        progressDialog.dismiss();
        setProgressStatus(refreshEntry.getGuid(), "Downloaded");

        unzip();
        processActiveLayerGeoJson();

        refreshList();
        refreshItemStatus();

        //AppUtil.unzip(savePath);
        //statusView.setText("READY");
    }

    @Override
    public void downloadFailed(String reason) {
        progressDialog.dismiss();

        GuiUtil.showError(this, reason);

    }

    @Override
    public void downloadProgress(double progress) {
        final int iProgress = (int)(progress);
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                //progressView.setProgress(iProgress);
                progressDialog.setProgress(iProgress);
            }
        });
    }


    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        /*
        if (requestCode == REQUEST_NEW_OFFLINE_MAP) {
            if (resultCode == RESULT_OK) {
                refreshList();
                refreshItemStatus();
            }
        }
        */
    }

    private class OfflineMapsAdapter extends ArrayAdapter<OfflineMapEntry> {

        private Context context;
        private List<OfflineMapEntry> offlineMaps;

        public OfflineMapsAdapter(Context context, List<OfflineMapEntry> offlineMaps) {
            //super(context, R.layout.attachment_item);
            super(context, android.R.layout.simple_list_item_1);

            this.context = context;
            this.offlineMaps = offlineMaps;
        }

        @Override
        public int getCount() {
            return offlineMaps.size();
        }

        @Override
        public View getView(int position, View convertView, ViewGroup parent) {

            View view = convertView;

            if (view == null) {
                LayoutInflater inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
                //view = inflater.inflate(R.layout.attachment_item2, parent, false);
                view = inflater.inflate(android.R.layout.simple_list_item_2, parent, false);
            }

            TextView firstLineView = (TextView)view.findViewById(android.R.id.text1);
            TextView secondLineView = (TextView)view.findViewById(android.R.id.text2);

            OfflineMapEntry entry = entries.get(position);

            firstLineView.setText(entry.getName());

            String description = "";
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd");
            String dateStr = sdf.format(entry.getPrepareDate());
            description += dateStr;
            if (entry.isPrepared()) {
                description += " - Ready";
            }
            else if(entry.getProgressStatus() == null) {
                description += " - Pending";
            }
            else {
                description += " - " + entry.getProgressStatus();
            }
            secondLineView.setText(description);

            return view;
        }

        @Override
        public OfflineMapEntry getItem(int position) {
            return offlineMaps.get(position);
        }

    }



}
