package org.quidity.demo.util;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.hardware.Camera;
import android.media.ExifInterface;
import android.os.Handler;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.Surface;
import android.view.View;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;


/**
 * Created by ferdiedanzfuss on 2014/09/04.
 */
public class GuiUtil {

    public static int MAX_TEXTURE_SIZE = 2048;  //Minimum required by all devices
    public static final int BUFFER_SIZE = 1024 * 8;

    private static final String TAG = GuiUtil.class.getCanonicalName();

    private static ProgressDialog progressDialog;

    public static void showError(Context context, String message) {

        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(message);
        builder.setTitle("Error");
        builder.setPositiveButton("OK", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                //dialogInterface.dismiss();
            }
        });
        AlertDialog dialog = builder.create();
        dialog.show();

    }

    public static void showWarning(Context context, String message) {

        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(message);
        builder.setTitle("Warning");
        builder.setPositiveButton("OK", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                //dialogInterface.dismiss();
            }
        });
        AlertDialog dialog = builder.create();
        dialog.show();

    }

    public static void focus(Context context, View view) {
        view.requestFocus();
        InputMethodManager imm = (InputMethodManager) context.getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.showSoftInput(view, InputMethodManager.SHOW_IMPLICIT);
    }

    /*
    public static void unfocus(Context context, View view) {
        InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
    }
    */

    public static int getScreenOrientation(Activity activity) {
        int rotation = activity.getWindowManager().getDefaultDisplay().getRotation();
        DisplayMetrics dm = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(dm);
        int width = dm.widthPixels;
        int height = dm.heightPixels;
        int orientation;
        // if the device's natural orientation is portrait:
        if ((rotation == Surface.ROTATION_0
                || rotation == Surface.ROTATION_180) && height > width ||
                (rotation == Surface.ROTATION_90
                        || rotation == Surface.ROTATION_270) && width > height) {
            switch(rotation) {
                case Surface.ROTATION_0:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                    break;
                case Surface.ROTATION_90:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                    break;
                case Surface.ROTATION_180:
                    orientation =
                            ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                    break;
                case Surface.ROTATION_270:
                    orientation =
                            ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                    break;
                default:
                    Log.e(TAG, "Unknown screen orientation. Defaulting to " +
                            "portrait.");
                    orientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                    break;
            }
        }
        // if the device's natural orientation is landscape or if the device
        // is square:
        else {
            switch(rotation) {
                case Surface.ROTATION_0:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                    break;
                case Surface.ROTATION_90:
                    orientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                    break;
                case Surface.ROTATION_180:
                    orientation =
                            ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                    break;
                case Surface.ROTATION_270:
                    orientation =
                            ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                    break;
                default:
                    Log.e(TAG, "Unknown screen orientation. Defaulting to " +
                            "landscape.");
                    orientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                    break;
            }
        }

        return orientation;
    }

    public static int calculateInSampleSize(
            BitmapFactory.Options options, int reqWidth, int reqHeight) {
        // Raw height and width of image
        final int height = options.outHeight;
        final int width = options.outWidth;
        int inSampleSize = 1;

        if (height > reqHeight || width > reqWidth) {

            final int halfHeight = height / 2;
            final int halfWidth = width / 2;

            // Calculate the largest inSampleSize value that is a power of 2 and keeps both
            // height and width larger than the requested height and width.
            while ((halfHeight / inSampleSize) > reqHeight
                    && (halfWidth / inSampleSize) > reqWidth) {
                inSampleSize *= 2;
            }
        }

        return inSampleSize;
    }

    public static Bitmap decodeSampledBitmapFromResource(Resources res, int resId,
                                                         int reqWidth, int reqHeight) {

        // First decode with inJustDecodeBounds=true to check dimensions
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeResource(res, resId, options);

        // Calculate inSampleSize
        options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);

        // Decode bitmap with inSampleSize set
        options.inJustDecodeBounds = false;
        return BitmapFactory.decodeResource(res, resId, options);
    }

    public static Bitmap decodeSampledBitmapFromByteArray(byte[] data, int reqWidth, int reqHeight) {

        // First decode with inJustDecodeBounds=true to check dimensions
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeByteArray(data, 0, data.length, options);

        // Calculate inSampleSize
        options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);

        // Decode bitmap with inSampleSize set
        options.inJustDecodeBounds = false;
        return BitmapFactory.decodeByteArray(data, 0, data.length, options);
    }

    public static Bitmap decodeSampledBitmapFromFile(String pathName, int reqWidth, int reqHeight) {

        // First decode with inJustDecodeBounds=true to check dimensions
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(pathName, options);

        // Calculate inSampleSize
        options.inSampleSize = calculateInSampleSize(options, reqWidth, reqHeight);

        // Decode bitmap with inSampleSize set
        options.inJustDecodeBounds = false;
        Bitmap bitmap = BitmapFactory.decodeFile(pathName, options);

        return bitmap;
    }

    public static Bitmap downscaleIfTooLargeForTexture(Bitmap bitmap) {
        //some devices have a max OpenGL texture limit of 2048x2048

        if (bitmap.getWidth() > MAX_TEXTURE_SIZE || bitmap.getHeight() > MAX_TEXTURE_SIZE) {

            //determine new width and height, keeping existing aspect ratio
            int originalWidth = bitmap.getWidth();
            int originalHeight = bitmap.getHeight();
            int newWidth = -1;
            int newHeight = -1;
            float multFactor = -1.0F;
            if(originalHeight > originalWidth) {
                newHeight = MAX_TEXTURE_SIZE;
                multFactor = (float) originalWidth/(float) originalHeight;
                newWidth = (int) (newHeight*multFactor);
            } else if(originalWidth > originalHeight) {
                newWidth = MAX_TEXTURE_SIZE;
                multFactor = (float) originalHeight/ (float)originalWidth;
                newHeight = (int) (newWidth*multFactor);
            } else if(originalHeight == originalWidth) {
                newHeight = MAX_TEXTURE_SIZE;
                newWidth = MAX_TEXTURE_SIZE;
            }

            Log.d(AppUtil.class.getCanonicalName(), "Bitmap " + originalWidth + "x" + originalHeight + " too large for OpenGL texture. Downscaling to " + newWidth + "x" + newHeight);

            bitmap = Bitmap.createScaledBitmap(bitmap, newWidth, newHeight, true);
        }

        return bitmap;
    }




   public static void fixRotationForCameraImage(Activity activity, String imagePath) {

       //int rotation = getRotationToBeAppliedToCameraImage(activity);
       int rotation = getImageExifRotation(imagePath);

       if (rotation == 0) {return;}

       Log.d(GuiUtil.class.getCanonicalName(), "Rotating camera image " + rotation + " degrees.");

       Matrix matrix = new Matrix();
       matrix.postRotate((float)rotation);

       Bitmap originalBitmap = BitmapFactory.decodeFile(imagePath);
       Bitmap rotatedBitmap = Bitmap.createBitmap(originalBitmap, 0, 0, originalBitmap.getWidth(), originalBitmap.getHeight(), matrix, false);

       //originalBitmap.recycle();

       saveBitmapToJpeg(rotatedBitmap, imagePath);

   }


    public static void saveBitmapToJpeg(Bitmap bitmap, String filePath) {
        try {
            File file = new File(filePath);
            file.createNewFile();
            FileOutputStream fos = new FileOutputStream(file);
            final BufferedOutputStream bos = new BufferedOutputStream(fos, BUFFER_SIZE);
            bitmap.compress(Bitmap.CompressFormat.JPEG, 100, bos);
            bos.flush();
            bos.close();
            fos.close();
        } catch (FileNotFoundException e) {
            Log.e(GuiUtil.class.getCanonicalName(), e.getLocalizedMessage(), e);
        } catch (IOException e) {
            Log.e(GuiUtil.class.getCanonicalName(), e.getLocalizedMessage(), e);
        }

    }

    public static int getImageExifRotation(String imagePath) {

        int rotate = 0;

        try {
            ExifInterface exif = new ExifInterface(imagePath);
            int exifOrientation = exif.getAttributeInt(
                    ExifInterface.TAG_ORIENTATION,
                    ExifInterface.ORIENTATION_NORMAL);

            switch (exifOrientation) {
                case ExifInterface.ORIENTATION_ROTATE_90:
                    rotate = 90;
                    break;

                case ExifInterface.ORIENTATION_ROTATE_180:
                    rotate = 180;
                    break;

                case ExifInterface.ORIENTATION_ROTATE_270:
                    rotate = 270;
                    break;
            }

        }
        catch (Exception e) {
            Log.d(GuiUtil.class.getCanonicalName(), e.getLocalizedMessage(), e);
        }

        return rotate;
    }

    /*
    public static int getRotationToBeAppliedToCameraImage(Activity activity) {

        int imageRotationDegrees = 0;

        //display rotation from natural position
        Display display = activity.getWindowManager().getDefaultDisplay();
        int displayRotation = display.getRotation();
        int displayRotationDegrees = 360 - getDisplayRotationDegrees(displayRotation); //converted to clockwise
        if (Math.abs(displayRotationDegrees) == 360) {displayRotationDegrees = 0;}
        Log.d(GuiUtil.class.getCanonicalName(), "Display rotation: " + displayRotationDegrees);

        //camera rotation from natural position
        try {
            Camera.CameraInfo info = new Camera.CameraInfo();
            Camera.getCameraInfo(0, info);
            int cameraRotationDegrees = info.orientation;
            Log.d(GuiUtil.class.getCanonicalName(), "Camera rotation: " + cameraRotationDegrees);

            imageRotationDegrees = displayRotationDegrees - cameraRotationDegrees;
            imageRotationDegrees = Math.abs(imageRotationDegrees);
            Log.d(GuiUtil.class.getCanonicalName(), "Image rotation: " + imageRotationDegrees);
        }
        catch (Exception e) {
            Log.e(GuiUtil.class.getCanonicalName(), e.getLocalizedMessage(), e);
        }

        return imageRotationDegrees;
    }


    private static int getDisplayRotationDegrees(int rotation) {
        int degrees = -1;

        if (rotation == Surface.ROTATION_0) {
            degrees = 0;
        }
        else if(rotation == Surface.ROTATION_90) {
            degrees = 90;
        }
        else if(rotation == Surface.ROTATION_180) {
            degrees = 180;
        }
        else if(rotation == Surface.ROTATION_270) {
            degrees = 270;
        }


        return degrees;
    }
    */


    /*
    public static void showProgress(final Context context, final String message) {

        Handler mainHandler = new Handler(context.getMainLooper());
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                progressDialog = new ProgressDialog(context);
                progressDialog.setIndeterminate(true);
                progressDialog.setMessage(message);
                progressDialog.show();
            }
        });

    }

    public static void hideProgress(final Context context) {
        Handler mainHandler = new Handler(context.getMainLooper());
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                if (progressDialog != null) {
                    progressDialog.hide();
                }
            }
        });

    }
    */
}
