package org.quidity.demo.activity;

import android.app.Activity;
import android.content.Intent;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.net.Uri;
import android.os.Bundle;
import android.os.Message;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.webkit.JavascriptInterface;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageView;
import android.widget.Toast;

import com.google.gson.Gson;

import org.json.JSONObject;

import java.text.AttributedCharacterIterator;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.quidity.demo.R;
import org.quidity.demo.Session;
import org.quidity.demo.Settings;
import org.quidity.demo.helper.FeatureEditTestHelper;
import org.quidity.demo.helper.LogoutHelper;
import org.quidity.demo.mapi.attributes.AttributesService;
import org.quidity.demo.mapi.attributes.Feature;
import org.quidity.demo.mapi.attributes.FeaturesRequest;
import org.quidity.demo.mapi.attributes.FeaturesResponse;
import org.quidity.demo.mapi.attributes.Geometry;
import org.quidity.demo.mapi.params.Layer;
import org.quidity.demo.mapi.params.ParameterService;
import org.quidity.demo.mapi.spatial.SpatialService;
import org.quidity.demo.mapi.spatial.UpdateFeatureRequest;
import org.quidity.demo.mapi.spatial.UpdateFeatureResponse;
import org.quidity.demo.mapi.tracking.AddTrackRequest;
import org.quidity.demo.mapi.tracking.AddTrackResponse;
import org.quidity.demo.mapi.tracking.TrackingService;
import org.quidity.demo.util.AppUtil;
import org.quidity.demo.util.GuiUtil;

import com.google.zxing.integration.android.IntentIntegrator;

public class OpenLayersMapActivity extends Activity implements SensorEventListener,
        LocationListener,
        AttributesService.FeaturesListener,
        SpatialService.UpdateFeatureListener,
        TrackingService.AddTrackListener
{

    public static final String EXTRA_ACTION = "action";
    public static final String EXTRA_LATITUDE = "latitude";
    public static final String EXTRA_LONGITUDE = "longitude";
    public static final String EXTRA_CAN_NAVIGATE = "CanNavigate";

    public static final int ACTION_NONE = 0;
    public static final int ACTION_MOVE_TO_POINT = 10;


    private static final int REQ_BASE_LAYER = 100;
    private static final int REQ_OTHER_LAYERS = 200;
    private static final int REQ_WORKSPACE = 300;
    private static final int REQ_ACTIVE_LAYER = 400;
    private static final int REQ_SEARCH = 500;

    private static final String GOOGLE_MAPS_PACKAGE = "com.google.android.apps.maps";
    private static final String GOOGLE_MAPS_ACTIVITY = "com.google.android.maps.MapsActivity";

    private WebView webView;
    private ImageView targetView;
    private String token;
    private String serial;
    private JSONObject jsLayers;

    //private SensorManager sensorManager;
    //private Sensor accelerometer;
    //private Sensor magnetometer;

    private float[] mGravity = new float[3];
    private float[] mGeomagnetic = new float[3];
    private float azimuth = 0f;

    private boolean rotationActive = false;
    private long lastRotationUpdate = 0;
    private boolean followMeActive = false;

    private LocationManager locationManager;

    private int currentZoomLevel;
    private Date lastLocationLogged = new Date();


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_open_layers_map);

        AppUtil.initActivity(this);

        //sensorManager = (SensorManager)getSystemService(SENSOR_SERVICE);
        //accelerometer = sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        //magnetometer = sensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD);

        targetView = (ImageView)findViewById(R.id.target);
        targetView.setVisibility(View.GONE);
        targetView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                onTargetClicked();
            }
        });

        locationManager = (LocationManager)getSystemService(LOCATION_SERVICE);

        loadMap();
    }

    private void loadMap() {
        webView = (WebView)findViewById(R.id.webView1);
        webView.getSettings().setJavaScriptEnabled(true);
        webView.getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        webView.getSettings().setSupportMultipleWindows(true);
        webView.addJavascriptInterface(Session.getSession().getJavascriptInterface(this), "Android");

        webView.setWebViewClient(new WebViewClient() {
            @Override
            public void onPageFinished(WebView view, String url) {
                super.onPageFinished(view, url);
                Log.d(getClass().getCanonicalName(), "Web page finished loading.");

                mapLoaded();
            }
        });

        MyWebChromeClient chromeClient = new MyWebChromeClient();
        webView.setWebChromeClient(chromeClient);

        //String data = AppUtil.readAsset(this, "map.html");
        //webView.loadData(data, "text/html", "UTF-8");

        //webView.loadUrl("http://www.mobilica.co.za/test.html");

        /*
        Layer baseLayer = Session.getSession().getCurrentBaseLayer();
        if((baseLayer.getId() != -1) && (baseLayer.getLayerType().startsWith("google"))) {
            //webView.loadUrl("http://www.mobilica.co.za/onemap/gmap.html");
            webView.loadUrl("file:///android_asset/gmap.html");
        }
        else {
            //webView.loadUrl("http://www.mobilica.co.za/onemap/map.html");
            webView.loadUrl("file:///android_asset/map.html");
        }
        */
        webView.loadUrl("file:///android_asset/map.html");
    }

    class MyWebChromeClient extends WebChromeClient {
        @Override
        public boolean onCreateWindow(WebView view, boolean isDialog, boolean isUserGesture, Message resultMsg) {
            boolean handled = false;
            if (isUserGesture) {
                WebView.HitTestResult result = view.getHitTestResult();
                if(result.getType() == WebView.HitTestResult.SRC_ANCHOR_TYPE) {
                    String data = result.getExtra();

                    //open with external browser
                    Uri uri = Uri.parse(data);
                    Intent intent = new Intent(Intent.ACTION_VIEW, uri);
                    startActivity(intent);

                    //handled = true;
                }
            }
            return handled;
        }
    }

    private void mapLoaded() {
        String serial = Session.getSession().getSerial();
        String token = Session.getSession().getToken();

        webView.loadUrl("javascript:initMap(" + serial + ", '" + token + "');");
        showCurrentLayer();

        int action = getIntent().getIntExtra(EXTRA_ACTION, ACTION_NONE);
        if (action == ACTION_MOVE_TO_POINT) {
            double latitude = getIntent().getDoubleExtra(EXTRA_LATITUDE, 0.0);
            double longitude = getIntent().getDoubleExtra(EXTRA_LONGITUDE, 0.0);

            zoomToPoint(latitude, longitude);
            showResultMarker(latitude, longitude);
        }
        else {

            OneMapJavascriptInterface jsint = Session.getSession().getJavascriptInterface(this);
            if (jsint.getCurrentZoomLevel() == 0) {
                zoomToLastKnownLocation();
            }
            else {
                //restore previous state
                setZoomLevel(jsint.getCurrentZoomLevel());
                zoomToPoint(jsint.getCurrentLatitude(), jsint.getCurrentLongitude());
            }

        }

    }

    private void zoomToLastKnownLocation() {
        Location location = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
        if (location != null) {
            zoomToPoint(location.getLatitude(), location.getLongitude());
        }
        else {
            Log.w(getClass().getCanonicalName(), "Could not get last known location.");
        }
    }

    private void showCurrentLayer() {

        webView.loadUrl("javascript:removeAllLayers();");

        showBaseLayer();
        showVisibleLayers();

    }

    private void showBaseLayer() {

        Layer layer = Session.getSession().getCurrentBaseLayer();

        if (layer.getId() == -1) {  //None base layer
            return;
        }

        if (layer.getLayerType().equalsIgnoreCase("osm")) {
            webView.loadUrl("javascript:addOSMLayer();");
        }
        else if (layer.getLayerType().equalsIgnoreCase("1map") || layer.getLayerType().equalsIgnoreCase("wms")) {
            String js = "javascript:addWMSLayer(" + layer.getId() + ", '" + layer.getAttribution() + "', " + layer.getBaseLayer() + ", " + layer.getMinScale() + ", " + layer.getMaxScale() + ", " + layer.getOpacity() + ");";
            webView.loadUrl(js);
        }
        else if (layer.getLayerType().startsWith("google")) {
            String js = "javascript:addGoogleLayer('" + layer.getLayerType() + "');";
            webView.loadUrl(js);
        }
        else {
            Toast.makeText(this, "Layer type " +  layer.getLayerType() + " under construction.", Toast.LENGTH_SHORT).show();
        }

    }

    private void showVisibleLayers() {

        List<Layer> layers = new ArrayList<Layer>();
        for(Layer layer : Session.getSession().getLayers()) {
            if (!layer.getBaseLayer()) {

                if (layer.isVisibility()) {

                    if (layer.getLayerType().equalsIgnoreCase("vector") || layer.getLayerType().equalsIgnoreCase("raster")) {
                        String js = "javascript:addWMSLayer(" + layer.getId() + ", '" + layer.getAttribution() + "', " + layer.getBaseLayer() + ", " + layer.getMinScale() + ", " + layer.getMaxScale() + ", " + layer.getOpacity() + ", '" + layer.getStyle() + "');";
                        webView.loadUrl(js);
                    }
                    else {
                        Toast.makeText(this, "Layer type " + layer.getLayerType() + " under construction.", Toast.LENGTH_SHORT).show();
                    }

                }

            }
        }


    }

    private void zoomToPoint(double latitude, double longitude) {
        //Toast.makeText(this, "Zoom to " + latitude + ", " + longitude, Toast.LENGTH_SHORT).show();

        String js = "javascript:zoomToPoint(" + latitude + ", " + longitude + ");";
        webView.loadUrl(js);

    }

    private void moveToPoint(double latitude, double longitude) {
        //Toast.makeText(this, "Zoom to " + latitude + ", " + longitude, Toast.LENGTH_SHORT).show();

        String js = "javascript:moveToPoint(" + latitude + ", " + longitude + ");";
        webView.loadUrl(js);

    }

    private void showMarker(double latitude, double longitude) {

        String js = "javascript:showMarker(" + latitude + ", " + longitude + ");";
        webView.loadUrl(js);

    }

    private void showResultMarker(double latitude, double longitude) {

        String js = "javascript:showResultMarker(" + latitude + ", " + longitude + ");";
        webView.loadUrl(js);

    }


    private void hideMarker() {

        String js = "javascript:hideMarker();";
        webView.loadUrl(js);

    }

    private void setZoomLevel(int newZoomLevel) {

        String js = "javascript:setZoomLevel(" + newZoomLevel + ");";
        webView.loadUrl(js);

        currentZoomLevel = newZoomLevel;
    }

    private void saveState() {

        String js = "javascript:saveState();";
        webView.loadUrl(js);

    }

    private void addVectorLayer(String geoJsonStr) {

        String js = "javascript:addVectorLayer('" + geoJsonStr + "');";
        webView.loadUrl(js);

    }


    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.open_layers_map, menu);
        return true;
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        MenuItem rotateMenu = menu.findItem(R.id.action_rotate);
        rotateMenu.setEnabled(followMeActive);
        rotateMenu.setChecked(rotationActive);

        MenuItem navigateItem = menu.findItem(R.id.action_navigate);
        //int action = getIntent().getIntExtra(EXTRA_ACTION, ACTION_NONE);
        //boolean canNavigate = (action == ACTION_MOVE_TO_POINT);
        boolean canNavigate = getIntent().getBooleanExtra(EXTRA_CAN_NAVIGATE, false);
        navigateItem.setEnabled(canNavigate);

        MenuItem featureInfoItem = menu.findItem(R.id.action_feature_info);
        List<Feature> selectedFeatures = Session.getSession().getSelectedFeatures();
        boolean featureInfoAvailable = selectedFeatures != null;
        featureInfoItem.setEnabled(featureInfoAvailable);

        MenuItem featureNewItem = menu.findItem(R.id.action_feature_new);
        boolean canEditFeature = false;
        boolean canEditSpatial = false;
        Layer activeLayer = Session.getSession().getCurrentActiveLayer();
        if (activeLayer != null) {
            canEditFeature = activeLayer.isEditMetadata();
            canEditSpatial = activeLayer.isEditSpatial();
        }
        featureNewItem.setEnabled(canEditFeature);

        MenuItem featureEditItem = menu.findItem(R.id.action_feature_edit);
        featureEditItem.setEnabled(featureInfoAvailable && canEditFeature);

        MenuItem featureMoveItem = menu.findItem(R.id.action_feature_move);
        featureMoveItem.setEnabled(featureInfoAvailable && canEditSpatial);

        return super.onPrepareOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        int id = item.getItemId();
        if (id == R.id.action_workspace) {
            showWorkspacesScreen();
            return true;
        }
        else if (id == R.id.action_base_layer) {
            showBaseLayersScreen();
            return true;
        }
        else if (id == R.id.action_other_layers) {
            showOtherLayersScreen();
            return true;
        }
        else if(id == R.id.action_active_layer) {
            showActiveLayerScreen();
            return true;
        }
        else if (id == R.id.action_search) {
            showSearchScreen();
            return true;
        }
        else if(id == R.id.action_logout) {
            logout();
            return true;
        }
        else if(id == R.id.action_follow) {
            toggleFollowMe();
            item.setChecked(followMeActive);
            return true;
        }
        else if(id == R.id.action_rotate) {
            toggleRotation();
            item.setChecked(rotationActive);
            return true;
        }
        else if(id == R.id.action_gps_info) {
            showGPSInfoScreen();
            return true;
        }
        else if(id == R.id.action_settings) {
            showSettingsScreen();
            return true;
        }
        else if(id == R.id.action_navigate) {
            navigate();
            return true;
        }
        else if(id == R.id.action_feature_info) {
            showFeatureInfo();
            return true;
        }
        else if (id == R.id.action_test_new) {
            Session.getSession().setEditMode(Session.EDIT_MODE_CREATE);

            testFeatureEdit();
            return true;
        }
        else if (id == R.id.action_test_edit) {
            Session.getSession().setEditMode(Session.EDIT_MODE_UPDATE);
            testFeatureEdit();

            //IntentIntegrator integrator = new IntentIntegrator(this);
            //integrator.initiateScan();

            return true;
        }
        else if(id == R.id.action_feature_new) {
            newFeature();

            return true;
        }
        else if(id == R.id.action_feature_edit) {
            editFeatureAttributes();

            return true;
        }
        else if(id == R.id.action_feature_move) {
            editFeatureGeometry();

            return true;
        }
        else if(id == R.id.action_coordinate) {
            showCoordinateScreen();

            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    private void showCoordinateScreen() {
        Intent intent = new Intent(this, CoordinateActivity.class);
        intent.setFlags(Intent.FLAG_ACTIVITY_NO_HISTORY);

        startActivityForResult(intent, REQ_SEARCH);
    }

    private void newFeature() {
        Session.getSession().setEditMode(Session.EDIT_MODE_CREATE);

        targetView.setVisibility(View.VISIBLE);
        Toast.makeText(this, "Pan the map to position, then tap the target to continue.", Toast.LENGTH_LONG).show();
    }

    private void onTargetClicked() {
        //will save current map centroid in JavaScriptInterface object in Session
        saveState();
        targetView.setVisibility(View.GONE);

        String editMode = Session.getSession().getEditMode();
        if (editMode.equalsIgnoreCase(Session.EDIT_MODE_UPDATE)) {
            updateGeometry();
        }
        else if (editMode.equalsIgnoreCase(Session.EDIT_MODE_CREATE)) {
            OneMapJavascriptInterface jsi = Session.getSession().getJavascriptInterface(this);
            Session.getSession().setEditMode(Session.EDIT_MODE_CREATE);
            Session.getSession().setCurrentLatitude(jsi.getCurrentLatitude());
            Session.getSession().setCurrentLongitude(jsi.getCurrentLongitude());

            Intent intent = new Intent(this, EditTemplateActivity.class);
            startActivity(intent);
        }
    }

    private void updateGeometry() {
        UpdateFeatureRequest request = new UpdateFeatureRequest();
        long layerId = Session.getSession().getCurrentActiveLayer().getId();
        long primaryId = AppUtil.getPrimaryIdForSelectedFeature();
        request.setLayerId(layerId);
        request.setPrimaryId(primaryId);

        Geometry geometry = new Geometry();
        geometry.setType("Point");
        OneMapJavascriptInterface jsi = Session.getSession().getJavascriptInterface(this);
        double latitude = jsi.getCurrentLatitude();
        double longitude = jsi.getCurrentLongitude();
        double[] coordinates = new double[] {longitude, latitude};
        geometry.setCoordinates(coordinates);

        SpatialService service = new SpatialService();
        service.updateFeature(request, geometry, this);
    }

    @Override
    public void updateFeatureCompleted(UpdateFeatureResponse response) {
        if (response.getMessage().equalsIgnoreCase("Success")) {
            Toast.makeText(this, "Location updated successfully.", Toast.LENGTH_SHORT).show();
        }
        else {
            GuiUtil.showError(this, response.getMessage());
        }
    }

    @Override
    public void updateFeatureFailed(String reason) {
        GuiUtil.showError(this, reason);
    }

    private void editFeatureAttributes() {
        Session.getSession().setEditMode(Session.EDIT_MODE_UPDATE);

        Intent intent = new Intent(this, EditTemplateActivity.class);
        startActivity(intent);
    }

    private void editFeatureGeometry() {
        Session.getSession().setEditMode(Session.EDIT_MODE_UPDATE);

        targetView.setVisibility(View.VISIBLE);
        Toast.makeText(this, "Pan the map to reposition, then tap the target to save.", Toast.LENGTH_LONG).show();
    }

    private void testFeatureEdit() {

        FeatureEditTestHelper helper = new FeatureEditTestHelper(this);
        helper.test();

    }

    private void showFeatureInfo() {
        Intent intent = new Intent(this, FeatureInfoPagerActivity.class);
        startActivity(intent);
    }

    private void navigate() {

        double latitude = getIntent().getDoubleExtra(EXTRA_LATITUDE, 0.0);
        double longitude = getIntent().getDoubleExtra(EXTRA_LONGITUDE, 0.0);

        //Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse("google.navigation:ll=-25.808611,28.256111"));
        String urlStr = "http://maps.google.com/maps?daddr=" + latitude + "," + longitude;
        Log.d(getClass().getCanonicalName(), urlStr);
        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(urlStr));

        boolean isGoogleMapsInstalled = AppUtil.isAppInstalled(this, GOOGLE_MAPS_PACKAGE);
        Log.d(getClass().getCanonicalName(), "isGoogleMapsInstalled: " + isGoogleMapsInstalled);
        if (isGoogleMapsInstalled) {
            intent.setClassName(GOOGLE_MAPS_PACKAGE, GOOGLE_MAPS_ACTIVITY);
        }
        startActivity(intent);
    }


    /*
    private void testCompass() {
        Intent intent = new Intent(this, CompassActivity.class);
        startActivity(intent);
    }
    */

    private void toggleFollowMe() {
        if (followMeActive) {
            stopFollowMe();
            stopRotation();
        }
        else {
            startFollowMe();
            startRotation();
        }
    }

    private void startFollowMe() {

        setZoomLevel(Settings.ZOOM_LEVEL_DEFAULT);

        locationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, 200, 0, this);

        followMeActive = true;

    }

    private void stopFollowMe() {
        locationManager.removeUpdates(this);
        hideMarker();

        followMeActive = false;
    }

    private void toggleRotation() {
        if (rotationActive) {
            stopRotation();
        }
        else {
            startRotation();
        }
    }

    private void startRotation() {
        //sensorManager.registerListener(this, accelerometer, SensorManager.SENSOR_DELAY_NORMAL);
        //sensorManager.registerListener(this, magnetometer, SensorManager.SENSOR_DELAY_NORMAL);

        rotationActive = true;

        //show marker north
        rotateMarker(0);

    }

    private void stopRotation() {
        //sensorManager.unregisterListener(this);

        //return to North at the top
        String js = "javascript:rotateTo(0);";
        webView.loadUrl(js);

        rotationActive = false;
    }

    private void showWorkspacesScreen() {
        saveState();
        Intent intent = new Intent(this, WorkspacesActivity.class);
        startActivityForResult(intent, REQ_WORKSPACE);
    }

    private void showBaseLayersScreen() {
        saveState();
        Intent intent = new Intent(this, BaseLayerActivity.class);
        startActivityForResult(intent, REQ_BASE_LAYER);
    }

    private void showOtherLayersScreen() {
        saveState();
        //Intent intent = new Intent(this, OtherLayersExpandableActivity.class);
        Intent intent = new Intent(this, OtherLayersMultiExpandableActivity.class);
        startActivityForResult(intent, REQ_OTHER_LAYERS);
    }

    private void showActiveLayerScreen() {
        saveState();
        Intent intent = new Intent(this, ActiveLayerActivity.class);
        startActivityForResult(intent, REQ_ACTIVE_LAYER);
    }

    private void showSearchScreen() {
        Intent intent = new Intent(this, SearchActivity.class);

        //startActivity(intent);
        intent.setFlags(Intent.FLAG_ACTIVITY_NO_HISTORY);

        startActivityForResult(intent, REQ_SEARCH);
    }

    private void showGPSInfoScreen() {
        Intent intent = new Intent(this, GPSInfoActivity.class);
        startActivity(intent);
    }

    private void showSettingsScreen() {
        Intent intent = new Intent(this, SettingsActivity.class);
        startActivity(intent);
    }

    private void logout() {
        LogoutHelper helper = new LogoutHelper(this);
        helper.initiateLogoutProcess();
    }

    @Override
    public void onSensorChanged(SensorEvent event) {
        //final float alpha = 0.97f;
        final float alpha = 0.7f;
        synchronized (this) {
            if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
                mGravity[0] = alpha * mGravity[0] + (1 - alpha)
                        * event.values[0];
                mGravity[1] = alpha * mGravity[1] + (1 - alpha)
                        * event.values[1];
                mGravity[2] = alpha * mGravity[2] + (1 - alpha)
                        * event.values[2];
            }
            if (event.sensor.getType() == Sensor.TYPE_MAGNETIC_FIELD) {
                mGeomagnetic[0] = alpha * mGeomagnetic[0] + (1 - alpha)
                        * event.values[0];
                mGeomagnetic[1] = alpha * mGeomagnetic[1] + (1 - alpha)
                        * event.values[1];
                mGeomagnetic[2] = alpha * mGeomagnetic[2] + (1 - alpha)
                        * event.values[2];
            }
            float R[] = new float[9];
            float I[] = new float[9];
            boolean success = SensorManager.getRotationMatrix(R, I, mGravity,
                    mGeomagnetic);
            if (success) {
                float orientation[] = new float[3];
                SensorManager.getOrientation(R, orientation);
                //azimuth = (float) Math.toDegrees(orientation[0]); // orientation
                azimuth = orientation[0]; // orientation
                //azimuth = (azimuth + 360) % 360;

                //int roundedAzimuth = (int)Math.round(azimuth);

                long now = System.currentTimeMillis();
                //only update every 200ms
                if (now - lastRotationUpdate > 200) {
                    String js = "javascript:rotateTo(" + azimuth + ");";
                    webView.loadUrl(js);
                    lastRotationUpdate = now;
                }

            }

        }
    }

    public void rotate(float degrees) {
        //float radials = (degrees + 360) % 360;
        double radians = Math.toRadians(degrees);

        long now = System.currentTimeMillis();
        //only update every 200ms
        if (now - lastRotationUpdate > 200) {
            String js = "javascript:rotateTo(" + radians + ");";
            webView.loadUrl(js);
            lastRotationUpdate = now;
        }

    }

    public void rotateMarker(float degrees) {
        long now = System.currentTimeMillis();
        //only update every 200ms
        if (now - lastRotationUpdate > 200) {
            String js = "javascript:rotateMarker(" + degrees + ");";
            webView.loadUrl(js);
            lastRotationUpdate = now;
        }

    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int i) {

    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {

        if (requestCode == REQ_BASE_LAYER) {
            //showCurrentLayer();
            loadMap();
        }
        else if(requestCode == REQ_OTHER_LAYERS) {
            //showCurrentLayer();
            loadMap();
        }
        else if (requestCode == REQ_WORKSPACE) {
            //showCurrentLayer();
            loadMap();
        }
        else if(requestCode == REQ_ACTIVE_LAYER) {
            //testFeatures();
        }
        else if(requestCode == REQ_SEARCH) {
            if(resultCode == RESULT_OK) {
                double latitude = data.getDoubleExtra(EXTRA_LATITUDE, 0.0);
                double longitude = data.getDoubleExtra(EXTRA_LONGITUDE, 0.0);

                getIntent().putExtra(EXTRA_CAN_NAVIGATE, true);
                getIntent().putExtra(EXTRA_LATITUDE, latitude);
                getIntent().putExtra(EXTRA_LONGITUDE, longitude);

                zoomToPoint(latitude, longitude);
                showResultMarker(latitude, longitude);
            }
        }

    }

    /*
    private void testFeatures() {
        AttributesService service = new AttributesService();
        FeaturesRequest request = new FeaturesRequest();
        Session session = Session.getSession();
        request.setLayerId(session.getCurrentActiveLayer().getId());
        request.setLatitude(-25.808611);
        request.setLongitude(28.256111);

        service.features(request, this);

    }
    */

    @Override
    public void featuresCompleted(FeaturesResponse response) {
        //GuiUtil.hideProgress(this);

        List<Feature> features = response.getItems().getFeatures();
        if ((features != null) && (features.size() > 0)) {
            Session.getSession().setSelectedFeatures(features);

            //Note not sure if we should highlight all features or only the frist feature
            //The features will all be on the same point, and there could be a lot of features

            //highlight the first feature
            Gson json = new Gson();
            Feature firstFeature = features.get(0);
            String geoJsonStr = json.toJson(firstFeature);
            addVectorLayer(geoJsonStr);
        }
        else {
            Session.getSession().setSelectedFeatures(null);
        }
    }

    @Override
    public void featuresFailed(String reason) {
        //GuiUtil.hideProgress(this);
        GuiUtil.showError(this, reason);
    }

    @Override
    public void onLocationChanged(Location location) {

        //adapt zoom level based on speed
        Settings settings = Settings.getSettings(this);
        if (settings.isAutoZoomEnabled()) {
            int zoomLevel = zoomLevelForSpeed(location.getSpeed());
            if (zoomLevel != currentZoomLevel) {
                setZoomLevel(zoomLevel);
                currentZoomLevel = zoomLevel;
                zoomToPoint(location.getLatitude(), location.getLongitude());
            }
        }

        moveToPoint(location.getLatitude(), location.getLongitude());
        showMarker(location.getLatitude(), location.getLongitude());

        if (rotationActive) {
            if (location.hasBearing()) {
                float bearing = location.getBearing();
                bearing = bearing * (-1);
                rotate(bearing);
            }
        }
        else {
            if (location.hasBearing()) {
                float bearing = location.getBearing();
                rotateMarker(bearing);
            }
        }

        if (settings.isLogLocation()) {
            Date now = new Date();
            long diff = now.getTime() - lastLocationLogged.getTime();
            long interval = settings.getLogLocationInterval() * 1000;    //convert to miliseconds
            if (diff >= interval) {
                logLocationToServer(location);
                lastLocationLogged = new Date();
            }


        }
    }

    private void logLocationToServer(Location location) {
        AddTrackRequest request = new AddTrackRequest();
        TrackingService service = new TrackingService();
        service.addTrack(request, location, this);
    }

    @Override
    public void addTrackCompleted(AddTrackResponse response) {
        if (response.getMessage().equalsIgnoreCase("Success")) {
            Log.d(getClass().getCanonicalName(), "Location logged to server.");
        }
        else {
            Log.d(getClass().getCanonicalName(), "COULD NOT LOG LOCATION TO SERVER - " + response.getMessage());
        }
    }

    @Override
    public void addTrackFailed(String reason) {
        Log.d(getClass().getCanonicalName(), "ERROR LOGGING LOCATION TO SERVER - " + reason);
    }

    private int zoomLevelForSpeed(float speedMps) {
        int zoomLevel = Settings.ZOOM_LEVEL_DEFAULT;

        float speedKmph = speedMps * 3.6f;

        if (speedKmph > Settings.SPEED_SLOW_DRIVE) {
            zoomLevel = Settings.ZOOM_LEVEL_SLOW_DRIVE;
        }
        else if (speedKmph > Settings.SPEED_FAST_DRIVE) {
            zoomLevel = Settings.ZOOM_LEVEL_FAST_DRIVE;
        }

        return zoomLevel;
    }

    public void mapClicked(double latitude, double longitude) {
        Log.d(getClass().getCanonicalName(), "Map clicked at " + latitude + ", " + longitude);

        Layer activeLayer = Session.getSession().getCurrentActiveLayer();
        if (activeLayer != null) {
            //GuiUtil.showProgress(this, "Please wait...");

            Log.d(getClass().getCanonicalName(), "Getting features at point for active layer: " + activeLayer.getCaption());
            AttributesService service = new AttributesService();
            FeaturesRequest request = new FeaturesRequest();
            request.setLayerId(activeLayer.getId());
            request.setLatitude(latitude);
            request.setLongitude(longitude);
            request.setShowAttachments(true);
            service.features(request, this);
        }
        else {
            Toast.makeText(this, "No Active Layer selected.", Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    public void onStatusChanged(String s, int i, Bundle bundle) {

    }

    @Override
    public void onProviderEnabled(String s) {

    }

    @Override
    public void onProviderDisabled(String s) {

    }

    @Override
    public void onBackPressed() {
        logout();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        stopRotation();
    }
}
