package org.quidity.demo.activity;

import android.app.Activity;
import android.graphics.Bitmap;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.animation.RotateAnimation;
import android.webkit.MimeTypeMap;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

import com.google.android.gms.maps.LocationSource;

import org.quidity.demo.R;
import org.quidity.demo.Session;
import org.quidity.demo.Settings;
import org.quidity.demo.mapi.attributes.AddAttachmentRequest;
import org.quidity.demo.mapi.attributes.AddAttachmentResponse;
import org.quidity.demo.mapi.attributes.Attachment;
import org.quidity.demo.mapi.attributes.AttachmentCameraPosition;
import org.quidity.demo.mapi.attributes.AttributesService;
import org.quidity.demo.mapi.attributes.Feature;
import org.quidity.demo.util.AppUtil;
import org.quidity.demo.util.GuiUtil;

import java.io.File;
import java.util.List;
import java.util.Set;

public class EditImageAttachmentActivity extends Activity implements AttributesService.AddAttachmentListener, LocationListener {

    public static final String EXTRA_FILE_PATH = "FILE_PATH";

    private EditText captionField;
    private EditText descriptionField;
    private TextView accuracyField;
    private ImageView imageView;

    private String filePath;
    private boolean attachmentLoaded;

    private LocationManager locationManager;
    private Location location;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_edit_image_attachment);

        AppUtil.initActivity(this);

        filePath = getIntent().getStringExtra(EXTRA_FILE_PATH);

        captionField = (EditText)findViewById(R.id.caption);
        descriptionField = (EditText)findViewById(R.id.description);
        accuracyField = (TextView)findViewById(R.id.accuracy);
        imageView = (ImageView)findViewById(R.id.image);

        locationManager = (LocationManager)getSystemService(LOCATION_SERVICE);

        if (Settings.getSettings(this).isCaptureImageLocation()) {
            accuracyField.setVisibility(View.VISIBLE);
            locationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, 0, 0, this);
        }
        else {
            accuracyField.setVisibility(View.GONE);
        }

    }


    @Override
    protected void onResume() {

        super.onResume();
    }

    @Override
    protected void onPause() {

        super.onPause();
    }



    @Override
    public void onLocationChanged(Location location) {
        this.location = location;

        if (location.hasAccuracy()) {
            String status = "Location Accuracy: " + Math.round(location.getAccuracy()) + "m";
            accuracyField.setText(status);
        }
        else {
            accuracyField.setText("Location Accuracy: Unknown");
        }
    }

    @Override
    public void onStatusChanged(String s, int i, Bundle bundle) {

    }

    @Override
    public void onProviderEnabled(String s) {

    }

    @Override
    public void onProviderDisabled(String s) {

    }




    @Override
    public void onWindowFocusChanged(boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);

        //we do it here because all views are now initialized and have correct sizes
        if (!attachmentLoaded) {
            //flag to ensure this happens only once
            attachmentLoaded = true;

            loadAttachment();
        }
    }



    private void loadAttachment() {
        Bitmap bitmap = GuiUtil.decodeSampledBitmapFromFile(filePath, imageView.getWidth(), imageView.getHeight());
        bitmap = GuiUtil.downscaleIfTooLargeForTexture(bitmap);
        imageView.setImageBitmap(bitmap);
    }

    private void save() {

        String caption = captionField.getText().toString().trim();
        String description = descriptionField.getText().toString().trim();

        if (caption.trim().length() == 0) {
            GuiUtil.showWarning(this, "Please enter a caption.");
            captionField.requestFocus();
            return;
        }

        Attachment params = new Attachment();
        params.setType("image");

        String extension = filePath.substring(filePath.lastIndexOf(".")+1);
        String mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
        params.setHeader(mimeType);

        params.setCaption(caption);
        if(description.length() > 0) {
            params.setDescription(description);
        }


        if (Settings.getSettings(this).isCaptureImageLocation()) {
            locationManager.removeUpdates(this);

            if (location == null) {
                GuiUtil.showWarning(this, "Location not available.");
                return;
            }

            if (location.hasAccuracy()) {
                int desiredAccuracy = Settings.getSettings(this).getDesiredAccuracy();
                if (location.getAccuracy() > desiredAccuracy) {
                    GuiUtil.showWarning(this, "Location not accurate enough. Should be at least " + desiredAccuracy + "m.");
                    return;
                }

                AttachmentCameraPosition position = new AttachmentCameraPosition();
                position.setLatitude(location.getLatitude());
                position.setLongitude(location.getLongitude());
                if (location.hasAltitude()) {
                    position.setAltitude(location.getAltitude());
                }

                double angle = calculateAngle();
                position.setAngle(angle);

                params.setCameraPosition(position);
            }
            else {
                GuiUtil.showWarning(this, "Location accuracy not available.");
                return;
            }
        }



        AttributesService service = new AttributesService();
        AddAttachmentRequest request = new AddAttachmentRequest();
        request.setLayerId(Session.getSession().getCurrentActiveLayer().getId());
        request.setPrimaryId(AppUtil.getPrimaryIdForSelectedFeature());
        service.addAttachment(request, params, filePath, this);

    }

    /**
     * Calculate angle between current location and feature location
     * @return
     */
    private double calculateAngle() {
        double angle = 0.0;
        try {
            Session session = Session.getSession();
            Feature feature = session.getSelectedFeatures().get(session.getSelectedFeatureIndex());
            List<Double> featureCoords = (List<Double>)feature.getGeometry().getCoordinates();
            double featureLongitude = featureCoords.get(0);
            double featureLatitude = featureCoords.get(1);

            Location featureLocation = new Location(LocationManager.GPS_PROVIDER);
            float bearing = location.bearingTo(featureLocation);
            angle = (double)bearing;
        }
        catch (Exception e) {
            Log.e(getClass().getCanonicalName(), e.getLocalizedMessage(), e);
        }

        return angle;
    }

    @Override
    public void addAttachmentCompleted(AddAttachmentResponse response) {
        Toast.makeText(this, response.getMessage(), Toast.LENGTH_SHORT).show();
        setResult(RESULT_OK);
        finish();
    }

    @Override
    public void addAttachmentFailed(String reason) {
        GuiUtil.showError(this, reason);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_edit_image_attachment, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        int id = item.getItemId();

        //noinspection SimplifiableIfStatement
        if (id == R.id.action_save) {
            save();
            return true;
        }

        return super.onOptionsItemSelected(item);
    }
}
