package za.co.karaboparking.siteinspection.fragment;


import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.location.Location;
import android.os.Bundle;
import android.provider.MediaStore;
import android.support.v4.app.Fragment;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.StaggeredGridLayoutManager;
import android.text.InputFilter;
import android.text.Spanned;
import android.text.method.DigitsKeyListener;
import android.util.Base64;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.DatePicker;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.TextView;
import android.widget.TimePicker;
import android.widget.Toast;

import com.google.firebase.database.DatabaseReference;
import com.google.firebase.database.FirebaseDatabase;
import com.squareup.picasso.Picasso;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import za.co.karaboparking.siteinspection.DB;
import za.co.karaboparking.siteinspection.MainActivity;
import za.co.karaboparking.siteinspection.R;
import za.co.karaboparking.siteinspection.Session;
import za.co.karaboparking.siteinspection.Settings;
import za.co.karaboparking.siteinspection.model.Choice;
import za.co.karaboparking.siteinspection.model.Inspection;
import za.co.karaboparking.siteinspection.model.InspectionEmailQueue;
import za.co.karaboparking.siteinspection.model.Photo;
import za.co.karaboparking.siteinspection.model.Question;
import za.co.karaboparking.siteinspection.model.Site;
import za.co.karaboparking.siteinspection.util.GUIUtil;

/**
 * A simple {@link Fragment} subclass.
 */
public class QuestionFragment extends Fragment implements View.OnClickListener {

    private static final String TAG = QuestionFragment.class.getCanonicalName();
    private static final int REQUEST_SELECT_PICTURE = 10;

    private Button nextButton;
    private Button previousButton;
    private TextView questionField;
    private TextView statusField;
    private Button addPhotoButton;
    private RadioGroup yesNoAnswerGroup;
    private RadioButton yesButton;
    private RadioButton noButton;
    private RadioGroup multiChoiceGroup;
    private EditText numericAnswerField;
    private EditText textAnswerField;
    private DatePicker dateAnswerField;
    private LinearLayout dateTimeAnswerGroup;
    private DatePicker dateTimeAnswerDateField;
    private TimePicker dateTimeAnswerTimeField;
    private LinearLayout amountAnswerGroup;
    private EditText amountAnswerField;
    private EditText commentsField;
    private Button cancelButton;
    private Button submitButton;
    private RecyclerView photosView;



    private QuestionFragmentListener listener;

    public QuestionFragment() {
        // Required empty public constructor
    }


    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        // Inflate the layout for this fragment
        View view = inflater.inflate(R.layout.fragment_question, container, false);

        questionField = (TextView)view.findViewById(R.id.question);
        nextButton = (Button)view.findViewById(R.id.next_button);
        previousButton = (Button)view.findViewById(R.id.previous_button);
        statusField = (TextView)view.findViewById(R.id.status);
        addPhotoButton = (Button)view.findViewById(R.id.add_photo_button);
        addPhotoButton.setOnClickListener(this);
        yesNoAnswerGroup = (RadioGroup)view.findViewById(R.id.yes_no_answer_group);
        yesButton = (RadioButton)view.findViewById(R.id.yes);
        noButton = (RadioButton)view.findViewById(R.id.no);
        multiChoiceGroup = (RadioGroup)view.findViewById(R.id.multi_choice_answer_group);
        numericAnswerField = (EditText)view.findViewById(R.id.numeric_answer);
        textAnswerField = (EditText)view.findViewById(R.id.text_answer);
        dateAnswerField = (DatePicker)view.findViewById(R.id.date_answer);
        dateTimeAnswerGroup = (LinearLayout)view.findViewById(R.id.date_time_answer_group);
        dateTimeAnswerDateField = (DatePicker)view.findViewById(R.id.date_time_answer_date);
        dateTimeAnswerTimeField = (TimePicker)view.findViewById(R.id.date_time_answer_time);
        amountAnswerGroup = (LinearLayout)view.findViewById(R.id.amount_answer_group);
        amountAnswerField = (EditText)view.findViewById(R.id.amount_answer);
        commentsField = (EditText)view.findViewById(R.id.comments);
        cancelButton = (Button)view.findViewById(R.id.cancel_button);
        submitButton = (Button)view.findViewById(R.id.submit_button);
        photosView = (RecyclerView)view.findViewById(R.id.photos);

        //initAmountAnswerField();

        nextButton.setOnClickListener(this);
        previousButton.setOnClickListener(this);
        cancelButton.setOnClickListener(this);
        submitButton.setOnClickListener(this);

        StaggeredGridLayoutManager layoutManager = new StaggeredGridLayoutManager(4, StaggeredGridLayoutManager.VERTICAL);
        photosView.setLayoutManager(layoutManager);

        refreshGUI();

        return view;
    }

    private void initAmountAnswerField() {

        amountAnswerField.setFilters(new InputFilter[] {
                new DigitsKeyListener(Boolean.FALSE, Boolean.TRUE) {
                    int beforeDecimal = 5, afterDecimal = 2;

                    @Override
                    public CharSequence filter(CharSequence source, int start, int end,
                                               Spanned dest, int dstart, int dend) {
                        String temp = amountAnswerField.getText() + source.toString();

                        if (temp.equals(".")) {
                            return "0.";
                        }
                        else if (temp.toString().indexOf(".") == -1) {
                            // no decimal point placed yet
                            if (temp.length() > beforeDecimal) {
                                return "";
                            }
                        } else {
                            temp = temp.substring(temp.indexOf(".") + 1);
                            if (temp.length() > afterDecimal) {
                                return "";
                            }
                        }

                        return super.filter(source, start, end, dest, dstart, dend);
                    }
                }
        });

    }


    @Override
    public void onAttach(Context context) {

        super.onAttach(context);
        if (context instanceof QuestionFragmentListener) {
            listener = (QuestionFragmentListener) context;
        } else {
            throw new RuntimeException(context.toString()
                    + " must implement QuestionFragmentListener");
        }

    }

    @Override
    public void onDetach() {

        super.onDetach();
        listener = null;

    }


    private void refreshGUI() {

        Session session = Session.getSession();
        int index = session.getQuestionIndex();
        Question question = session.getInspection().getQuestionnaire().getQuestions().get(index);

        questionField.setText(question.getName());

        yesNoAnswerGroup.setVisibility(View.GONE);
        multiChoiceGroup.setVisibility(View.GONE);
        numericAnswerField.setVisibility(View.GONE);
        textAnswerField.setVisibility(View.GONE);
        dateTimeAnswerGroup.setVisibility(View.GONE);
        amountAnswerGroup.setVisibility(View.GONE);
        dateAnswerField.setVisibility(View.GONE);

        if (question.getType() == Question.QuestionType.YesNo) {

            yesNoAnswerGroup.setVisibility(View.VISIBLE);

            if (question.getBoolAnswer() == null) {
                yesNoAnswerGroup.clearCheck();
            } else if (question.getBoolAnswer() == true) {
                yesButton.setChecked(true);
            } else if (question.getBoolAnswer() == false) {
                noButton.setChecked(true);
            }

            GUIUtil.hideKeyboard(getActivity());
            yesNoAnswerGroup.setFocusable(true);
            yesNoAnswerGroup.setFocusableInTouchMode(true);
            yesNoAnswerGroup.requestFocus();

        }
        else if (question.getType() == Question.QuestionType.MultiChoice) {
            multiChoiceGroup.setVisibility(View.VISIBLE);
            multiChoiceGroup.removeAllViews();
            for (Choice choice : question.getChoices()) {
                RadioButton rb = new RadioButton(getContext());
                rb.setId(rb.generateViewId());
                rb.setText(choice.getName());
                rb.setTextSize(22);
                rb.setPadding(15,15,15,15);
                if (choice.getName().equalsIgnoreCase(question.getTextAnswer())) {
                    rb.setChecked(true);
                }
                multiChoiceGroup.addView(rb);
            }

            GUIUtil.hideKeyboard(getActivity());
            multiChoiceGroup.setFocusable(true);
            multiChoiceGroup.setFocusableInTouchMode(true);
            multiChoiceGroup.requestFocus();

        }
        else if (question.getType() == Question.QuestionType.Number) {

            numericAnswerField.setVisibility(View.VISIBLE);

            if (question.getIntAnswer() == null) {
                numericAnswerField.setText(null);
            }
            else {
                String answerStr = Integer.toString(question.getIntAnswer());
                numericAnswerField.setText(answerStr);
            }

            numericAnswerField.requestFocus();
            GUIUtil.showKeyboard(getActivity(), numericAnswerField);

        }
        else if (question.getType() == Question.QuestionType.Text) {

            textAnswerField.setVisibility(View.VISIBLE);
            textAnswerField.setText(question.getTextAnswer());

            textAnswerField.requestFocus();
            GUIUtil.showKeyboard(getActivity(), textAnswerField);

        }
        else if (question.getType() == Question.QuestionType.DateTime) {
            dateTimeAnswerGroup.setVisibility(View.VISIBLE);

            Date date = null;
            String dateStr = question.getDateAnswer();
            if (dateStr != null) {
                try {
                    date = new SimpleDateFormat(Settings.DATETIME_FORMAT).parse(dateStr);
                } catch (ParseException e) {
                    Log.e(TAG, e.getMessage(), e);
                }
            }

            if(date != null) {
                Calendar cal = Calendar.getInstance();
                cal.setTime(date);
                int year = cal.get(Calendar.YEAR);
                int month = cal.get(Calendar.MONTH);
                int dayOfMonth = cal.get(Calendar.DAY_OF_MONTH);
                dateTimeAnswerDateField.updateDate(year, month, dayOfMonth);

                int hour = cal.get(Calendar.HOUR_OF_DAY);
                int minute = cal.get(Calendar.MINUTE);
                dateTimeAnswerTimeField.setHour(hour);
                dateTimeAnswerTimeField.setMinute(minute);
            }

            GUIUtil.hideKeyboard(getActivity());
            dateTimeAnswerGroup.setFocusable(true);
            dateTimeAnswerGroup.setFocusableInTouchMode(true);
            dateTimeAnswerGroup.requestFocus();

        }
        else if (question.getType() == Question.QuestionType.Date) {
            dateAnswerField.setVisibility(View.VISIBLE);

            Date date = null;
            String dateStr = question.getDateAnswer();
            if (dateStr != null) {
                try {
                    date = new SimpleDateFormat(Settings.DATE_FORMAT).parse(dateStr);
                } catch (ParseException e) {
                    Log.e(TAG, e.getMessage(), e);
                }
            }

            if(date != null) {
                Calendar cal = Calendar.getInstance();
                cal.setTime(date);
                int year = cal.get(Calendar.YEAR);
                int month = cal.get(Calendar.MONTH);
                int dayOfMonth = cal.get(Calendar.DAY_OF_MONTH);
                dateAnswerField.updateDate(year, month, dayOfMonth);
            }

            GUIUtil.hideKeyboard(getActivity());
            dateAnswerField.setFocusable(true);
            dateAnswerField.setFocusableInTouchMode(true);
            dateAnswerField.requestFocus();

        }
        else if (question.getType() == Question.QuestionType.Amount) {

            amountAnswerGroup.setVisibility(View.VISIBLE);

            if (question.getDecimalAnswer() == null) {
                amountAnswerField.setText(null);
            }
            else {
                String answerStr = Double.toString(question.getDecimalAnswer());
                amountAnswerField.setText(answerStr);
            }

            amountAnswerField.requestFocus();
            GUIUtil.showKeyboard(getActivity(), amountAnswerField);

        }


        commentsField.setText(question.getComments());

        if (question.getCommentsDescription() != null) {
            commentsField.setHint(question.getCommentsDescription());
        }
        else {
            commentsField.setHint(null);
        }

        int questionNo = index + 1;
        //int total = session.getQuestions().size();
        int total = session.getInspection().getQuestionnaire().getQuestions().size();
        String status = questionNo + " of " + total;
        statusField.setText(status);

        int maxIndex = session.getInspection().getQuestionnaire().getQuestions().size() - 1;
        if (index >= maxIndex) {
            nextButton.setEnabled(false);
            nextButton.setAlpha(0.5f);
        }
        else {
            nextButton.setEnabled(true);
            nextButton.setAlpha(1.0f);
        }

        if (index <= 0) {
            previousButton.setEnabled(false);
            previousButton.setAlpha(0.5f);
        }
        else {
            previousButton.setEnabled(true);
            previousButton.setAlpha(1.0f);
        }

        PhotoAdapter adapter = new PhotoAdapter(question.getPhotos());
        photosView.setAdapter(adapter);
    }

    @Override
    public void onClick(View view) {

        if (view == nextButton) {
            showNextQuestion();
        }
        else if (view == previousButton) {
            showPreviousQuestion();
        }
        else if (view == addPhotoButton) {
            addPhoto();
        }
        else if (view == cancelButton) {
            cancelAction();
        }
        else if (view == submitButton) {
            submit();
        }

    }

    private void submit() {

        //first validate and save current question
        if (!validateCurrentQuestion()) {
            return;
        }
        save();

        if (validateInspection()) {

            //Upload location
            MainActivity activity = (MainActivity)getActivity();
            activity.stopLocationUpdates();
            Location location = activity.getLocation();
            if (location != null) {
                Inspection inspection = Session.getSession().getInspection();
                inspection.setLatitude(location.getLatitude());
                inspection.setLongitude(location.getLongitude());
                inspection.setAccuracy(location.getAccuracy());

                completeSubmit();
            }
            else {
                //Toast.makeText(getContext(), "WARNING: Could not determine your location!", Toast.LENGTH_LONG).show();
                confirmSubmitWithoutLocation();
            }


        }

    }

    private void confirmSubmitWithoutLocation() {

        AlertDialog.Builder dialog = new AlertDialog.Builder(getActivity());
        dialog.setTitle("GPS Location Not Available");
        dialog.setMessage("Could not determine your location. You can continue, but note that a map will not be available on the report.");
        dialog.setNegativeButton("Cancel", null);
        dialog.setPositiveButton("Continue", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                completeSubmit();
            }
        });
        dialog.show();

    }

    private void completeSubmit() {

        GUIUtil.showProgress("Submitting Inspection...");
        saveInspectionToDb();
        updateSiteInfoToDb();
        addInspectionToEmailQueue();
        GUIUtil.hideProgress();

        GUIUtil.showMessage(getContext(), "Inspection Completed", "Thank you for submitting the site inspection!");
        getActivity().getSupportFragmentManager().popBackStack("Sites", 0);

    }

    private boolean validateInspection() {

        Inspection inspection = Session.getSession().getInspection();

        for (Question question : inspection.getQuestionnaire().getQuestions()) {
            if (question.getType() == Question.QuestionType.YesNo) {
                if (question.getBoolAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.MultiChoice) {
                if (question.getTextAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.Number) {
                if (question.getIntAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.Text) {
                if (question.getTextAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.DateTime) {
                if (question.getDateAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.Date) {
                if (question.getDateAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }
            else if (question.getType() == Question.QuestionType.Amount) {
                if (question.getDecimalAnswer() == null) {
                    GUIUtil.showWarning(getContext(), "You have not completed all the questions.");
                    return false;
                }
            }

        }

        return true;

    }

    private void saveInspectionToDb() {

        FirebaseDatabase db = FirebaseDatabase.getInstance();
        Inspection inspection = Session.getSession().getInspection();
        DatabaseReference ref = db.getReference(DB.INSPECTIONS)
                .child(inspection.getSiteCode())
                .child(inspection.getInspectionDate());
        ref.setValue(inspection);

    }

    private void addInspectionToEmailQueue() {

        Inspection inspection = Session.getSession().getInspection();
        Site site = Session.getSession().getSite();

        InspectionEmailQueue queue = new InspectionEmailQueue();
        String dateAddedStr = new Date().toString();
        queue.setDateAddedToQueue(dateAddedStr);
        queue.setSiteCode(inspection.getSiteCode());
        queue.setInspectionDate(inspection.getInspectionDate());
        queue.setEmailTo(inspection.getUserName());
        queue.setSiteName(site.getName());

        FirebaseDatabase db = FirebaseDatabase.getInstance();
        DatabaseReference ref = db.getReference(DB.INSPECTION_EMAIL_QUEUE);
        ref.push().setValue(queue);

    }


    private void updateSiteInfoToDb() {

        Inspection inspection = Session.getSession().getInspection();
        Site site = Session.getSession().getSite();

        Date date = new Date();

        String dateStr = new SimpleDateFormat(Settings.DATE_FORMAT).format(date);
        site.setLastInspectionDate(dateStr);

        site.setLastInspectionUserId(inspection.getUserId());
        site.setLastInspectionUserName(inspection.getUserName());

        FirebaseDatabase db = FirebaseDatabase.getInstance();
        DatabaseReference ref = db.getReference("Sites/" + site.getCode());

        ref.setValue(site);

    }

    private void cancelAction() {

        AlertDialog.Builder builder = new AlertDialog.Builder(getContext());
        builder.setTitle("Cancel Questionnaire");
        builder.setMessage("You will loose all your changes. Are you sure you want to cancel?");
        builder.setPositiveButton("Yes", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {

                GUIUtil.hideKeyboard(getActivity());

                //Stop location updates
                MainActivity activity = (MainActivity)getActivity();
                activity.stopLocationUpdates();

                //go back
                getActivity().getSupportFragmentManager().popBackStack();
            }
        });
        builder.setNegativeButton("No", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                //do nothing
            }
        });
        builder.show();
    }

    public void addPhoto() {

        //Save current state
        save();

//        Intent pickIntent = new Intent();
//        pickIntent.setType("image/*");
//        pickIntent.setAction(Intent.ACTION_GET_CONTENT);
//
//        Intent takePhotoIntent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
//
//        String pickTitle = "How would you like to add the photo?";
//        Intent chooserIntent = Intent.createChooser(pickIntent, pickTitle);
//        chooserIntent.putExtra
//                (
//                        Intent.EXTRA_INITIAL_INTENTS,
//                        new Intent[] { takePhotoIntent }
//                );
//
//        startActivityForResult(chooserIntent, REQUEST_SELECT_PICTURE);

        Intent takePhotoIntent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
        startActivityForResult(takePhotoIntent, REQUEST_SELECT_PICTURE);


    }


    private void showNextQuestion() {

        if (!validateCurrentQuestion()) {
            return;
        }

        save();

        Session session = Session.getSession();

        int currentIndex = session.getQuestionIndex();
        int nextIndex = currentIndex + 1;
        int maxIndex = session.getInspection().getQuestionnaire().getQuestions().size() - 1;

        if (nextIndex > maxIndex) {
            nextIndex = maxIndex;
        }

        session.setQuestionIndex(nextIndex);

        refreshGUI();

    }

    private void showPreviousQuestion() {

        //Note we only validate when we go forward, not back

        save();

        Session session = Session.getSession();

        int currentIndex = session.getQuestionIndex();
        int nextIndex = currentIndex - 1;

        if (nextIndex < 0) {
            nextIndex = 0;
        }

        session.setQuestionIndex(nextIndex);

        refreshGUI();

    }

    private static final String COMMENTS_REQUIRED_WARNING = "Please fill in the comments field.";
    private static final String PHOTOS_REQUIRED_WARNING = "Please add a photo.";

    private boolean validateCurrentQuestion() {

        boolean valid = true;

        Question question = currentQuestion();

        //Check if required comments and photos are provided
        String comments = commentsField.getText().toString().trim();
        boolean haveComments = comments.length() > 0;
        boolean havePhotos = question.getPhotos().size() > 0;


        if (question.isCommentsRequired() != null && question.isCommentsRequired()) {
            if (!haveComments) {
                GUIUtil.showWarning(getContext(), COMMENTS_REQUIRED_WARNING);
                return false;
            }
        }

        if (question.isPhotoRequired() != null && question.isPhotoRequired()) {
            if (!havePhotos) {
                GUIUtil.showWarning(getContext(), PHOTOS_REQUIRED_WARNING);
                return false;
            }
        }


        return valid;

    }

    private void save() {

        Question question = currentQuestion();

        if (question.getType() == Question.QuestionType.YesNo) {

            if (yesButton.isChecked() || noButton.isChecked()) {
                question.setBoolAnswer(yesButton.isChecked());
            } else {
                question.setBoolAnswer(null);
            }

        }
        else if (question.getType() == Question.QuestionType.MultiChoice) {
            int resId = multiChoiceGroup.getCheckedRadioButtonId();
            if (resId == -1) {
                question.setTextAnswer(null);
            }
            else {
                //RadioButton rb = (RadioButton)getView().findViewById(resId);
                RadioButton rb = (RadioButton) multiChoiceGroup.findViewById(resId);
                if (rb == null) {
                    Log.e(TAG, "This should not happen!");
                }
                else {
                    String value = rb.getText().toString();
                    question.setTextAnswer(value);
                }
            }
        }
        else if (question.getType() == Question.QuestionType.Number) {

            String answerStr = numericAnswerField.getText().toString().trim();
            if (answerStr.length() == 0) {
                question.setIntAnswer(null);
            }
            else {
                try {
                    int intAnswer = Integer.parseInt(answerStr);
                    question.setIntAnswer(intAnswer);
                }
                catch (NumberFormatException nfe) {
                    question.setIntAnswer(null);
                    GUIUtil.showWarning(getContext(), "Invalid answer. Must be an integer number.");
                }
            }
        }
        else if (question.getType() == Question.QuestionType.Text) {

            String answerStr = textAnswerField.getText().toString().trim();
            if (answerStr.length() == 0) {
                question.setTextAnswer(null);
            }
            else {
                question.setTextAnswer(answerStr);
            }
        }
        else if (question.getType() == Question.QuestionType.DateTime) {

            int year = dateTimeAnswerDateField.getYear();
            int month = dateTimeAnswerDateField.getMonth();
            int dayOfMonth = dateTimeAnswerDateField.getDayOfMonth();
            int hour = dateTimeAnswerTimeField.getHour();
            int minute = dateTimeAnswerTimeField.getMinute();

            Calendar cal = Calendar.getInstance();
            cal.set(year, month, dayOfMonth, hour, minute);
            Date date = cal.getTime();

            String dateStr = new SimpleDateFormat(Settings.DATETIME_FORMAT).format(date);
            question.setDateAnswer(dateStr);

        }
        else if (question.getType() == Question.QuestionType.Date) {

            int year = dateAnswerField.getYear();
            int month = dateAnswerField.getMonth();
            int dayOfMonth = dateAnswerField.getDayOfMonth();

            Calendar cal = Calendar.getInstance();
            cal.set(year, month, dayOfMonth);
            Date date = cal.getTime();

            String dateStr = new SimpleDateFormat(Settings.DATE_FORMAT).format(date);
            question.setDateAnswer(dateStr);

        }
        else if (question.getType() == Question.QuestionType.Amount) {

            String answerStr = amountAnswerField.getText().toString().trim();
            if (answerStr.length() == 0) {
                question.setDecimalAnswer(null);
            }
            else {
                try {
                    double decimalAnswer = Double.parseDouble(answerStr);
                    question.setDecimalAnswer(decimalAnswer);
                }
                catch (NumberFormatException nfe) {
                    question.setIntAnswer(null);
                    GUIUtil.showWarning(getContext(), "Invalid answer. Must be a decimal number.");
                }
            }
        }

        String comments = commentsField.getText().toString().trim();
        question.setComments(comments);

    }

    private Question currentQuestion() {
        Session session = Session.getSession();
        int index = session.getQuestionIndex();
        Question question = session.getInspection().getQuestionnaire().getQuestions().get(index);
        return question;
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == REQUEST_SELECT_PICTURE) {

            if (resultCode == Activity.RESULT_OK) {

                //get a small thumbnail image
                Bitmap thumbnailImage = (Bitmap) data.getExtras().get("data");

                //InputStream inputStream = context.getContentResolver().openInputStream(data.getData());

                savePhoto(thumbnailImage);
                refreshGUI();
            }

        }

        super.onActivityResult(requestCode, resultCode, data);


    }

    private void savePhoto(Bitmap thumbnailImage) {

        try {

            Log.d(TAG, "Photo dimensions: " + thumbnailImage.getWidth() + "x" + thumbnailImage.getHeight());
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            thumbnailImage.compress(Bitmap.CompressFormat.JPEG, 100, baos);
            byte[] data = baos.toByteArray();
            Log.d(TAG, "Photo size: " + data.length + " bytes");

            String encodedStr = Base64.encodeToString(data, Base64.DEFAULT);

            Photo photo = new Photo();
            photo.setThumbnailImage(encodedStr);

            Question question = currentQuestion();
            question.getPhotos().add(photo);

        }
        catch (Exception e) {
            Log.e(TAG, e.getMessage(), e);
            GUIUtil.showWarning(getContext(), e.getMessage());
        }

    }


    class PhotoViewHolder extends RecyclerView.ViewHolder {

        private ImageView imageView;

        public PhotoViewHolder(View itemView) {
            super(itemView);
            imageView = (ImageView) itemView.findViewById(R.id.image);
        }

        public ImageView getImageView() {
            return imageView;
        }
    }

    class PhotoAdapter extends RecyclerView.Adapter<PhotoViewHolder> {

        private List<Photo> photos;

        public PhotoAdapter(List<Photo> photos) {
            this.photos = photos;
        }

        @Override
        public PhotoViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
            LayoutInflater inflater = getActivity().getLayoutInflater();
            View itemView = inflater.inflate(R.layout.photo_thumbnail_item, parent, false);
            //itemView.setOnClickListener(QuestionFragment.this);
            PhotoViewHolder holder = new PhotoViewHolder(itemView);
            return holder;
        }

        @Override
        public void onBindViewHolder(PhotoViewHolder holder, int position) {
            Photo photo = photos.get(position);
            String encodedStr = photo.getThumbnailImage();
            byte[] data = Base64.decode(encodedStr, Base64.DEFAULT);
            Bitmap bitmap = BitmapFactory.decodeByteArray(data, 0, data.length);
            holder.getImageView().setImageBitmap(bitmap);
        }

        @Override
        public int getItemCount() {
            return photos.size();
        }

    }



    public interface QuestionFragmentListener {

    }

}
